'--------------------------------------------------------------------------------------------------+
'- License Stuff                                                                                   |
'-                                                                                                 |
'-                                                                                                 |
'-   SPFLite is free software: you can redistribute it and/or modify                               |
'-   it under the terms of the GNU General Public License as published by                          |
'-   the Free Software Foundation, either version 3 of the License, or                             |
'-   (at your option) any later version.                                                           |
'-                                                                                                 |
'-   SPFLite is distributed in the hope that it will be useful,                                    |
'-   but WITHOUT ANY WARRANTY; without even the implied warranty of                                |
'-   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                 |
'-   GNU General Public License for more details.                                                  |
'-                                                                                                 |
'-   You should have received a copy of the GNU General Public License                             |
'-   along with SPFLite.  If not, see <https://www.gnu.org/licenses/>.                             |
'-                                                                                                 |
'--------------------------------------------------------------------------------------------------+
'- FMPCmd.inc                                                                                      |
'--------------------------------------------------------------------------------------------------+
                                                                  '
'--------------------------------------------------------------------------------------------------+
'- File Manager Primary Commands                                                                   |
'--------------------------------------------------------------------------------------------------+


METHOD FMPCmdAll(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- ALL FM Command                                                                                  |
'--------------------------------------------------------------------------------------------------+
LOCAL lCmd, MEditCmd AS STRING, i AS LONG, t, tt AS STRING        '
LOCAL hFile AS LONG, hfile2 AS DWORD, DefCtr AS LONG              '
LOCAL CreationTime, LastAccessTime, LastWriteTime AS FILETIME     '
LOCAL SystemTimeArea AS SYSTEMTIME                                '
DIM Deferred(1 TO gFMDCtr) AS STRING                              '
   MEntry                                                         '
   lCmd = GetNextWord(pCmd, %Strip)                               ' Strip off the ALL command
   lCmd = GetNextWord(pCmd, %Strip)                               ' Strip off the command to be performed

   '-----------------------------------------------------------------------------------------------+
   '- Split based on type of requested command                                                     |
   '-----------------------------------------------------------------------------------------------+
   SELECT CASE AS CONST$ UUCASE(lCmd)                             '

      '--------------------------------------------------------------------------------------------+
      '- ALL BROWSE                                                                                |
      '--------------------------------------------------------------------------------------------+
      CASE "B", "BRO", "BROWSE"                                   '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               me.DoBROWSE(i)                                     ' Go do Browse
            END IF                                                '
         NEXT i                                                   '

      '--------------------------------------------------------------------------------------------+
      '- ALL CLONE                                                                                 |
      '--------------------------------------------------------------------------------------------+
      CASE "C", "CLONE"                                           '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               me.CallTab("CLONE", $DQ + gFMD(i).FullPath + $DQ)  '
            END IF                                                '
         NEXT i                                                   '

      '--------------------------------------------------------------------------------------------+
      '- ALL CANCEL                                                                                |
      '--------------------------------------------------------------------------------------------+
      CASE "CAN", "CANCEL"                                        '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FOpen) THEN                          ' Only look at OPEN entries
               INCR DefCtr                                        ' Save it
               Deferred(DefCtr) = FORMAT$(gFMD(i).Uniq, "000") + " CANCEL" '
            END IF                                                '
         NEXT i                                                   '

      '--------------------------------------------------------------------------------------------+
      '- ALL Edit                                                                                  |
      '--------------------------------------------------------------------------------------------+
      CASE "E", "EDIT", "S", "SEL", "SELECT"                      '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               me.DoEDIT(i)                                       ' Go do Browse
            END IF                                                '
         NEXT i                                                   '

      '--------------------------------------------------------------------------------------------+
      '- ALL END                                                                                   |
      '--------------------------------------------------------------------------------------------+
      CASE "END"                                                  '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FOpen) THEN                          ' Only look at OPEN entries
               INCR DefCtr                                        ' Save it
               Deferred(DefCtr) = FORMAT$(gFMD(i).Uniq, "000") + " END" '
            END IF                                                '
         NEXT i                                                   '

      '--------------------------------------------------------------------------------------------+
      '- ALL MEDIT                                                                                 |
      '--------------------------------------------------------------------------------------------+
      CASE "M", "MEDIT"                                           '
         FOR i = 1 TO gFMDCtr                                     ' Lets look asth the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               IF ISNULL(MEditCmd) THEN                           ' First M command?
                  MEditCmd = $DQ + gFMD(i).FullPath + $DQ + " "   '
                  MEditCmd += " NEW "                             '
               ELSE                                               ' Else just add a filename on
                  MEditCmd += $DQ + gFMD(i).FullPath + $DQ + " "  '
               END IF                                             '
            END IF                                                '
         NEXT i                                                   '
         me.CallTab("MEDIT", MEditCmd)                            '

      '--------------------------------------------------------------------------------------------+
      '- ALL PRINT                                                                                 |
      '--------------------------------------------------------------------------------------------+
      CASE "P", "PRINT"                                           '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               me.FMLCmdPrintIt($DQ + gFMD(i).FullPath + $DQ, 0)  '
            END IF                                                '
         NEXT i                                                   '
         MErrExit(0, "All files Printed")                         '

      '--------------------------------------------------------------------------------------------+
      '- ALL TOUCH                                                                                 |
      '--------------------------------------------------------------------------------------------+
      CASE "T", "TOUCH"                                           '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               TRY                                                ' Ignore errors
                  hFile = FREEFILE                                ' Touch the file now
                  OPEN gFMD(i).FullPath FOR APPEND ACCESS READ WRITE LOCK SHARED AS #hFile   '
                  hFile2 = FILEATTR(hFile,2)                      ' Get Windows handle for the file
                  GetFileTime hFile2, CreationTime, LastAccessTime, LastWriteTime   ' Fetch timestamps
                  GetSystemTime SystemTimeArea                    ' Get the current time
                  SystemtimeToFileTime SystemTimeArea, LastWriteTime ' Put it in LastWriteTime
                  SetFileTime hFile2, CreationTime, LastAccessTime, LastWriteTime   '
               CATCH                                              '
               END TRY                                            '
               CLOSE #hFile                                       ' Close it
            END IF                                                '
         NEXT i                                                   '
         DoSet(%LoadData)                                         ' Ask for refresh
         MErrExit(0, "All files Touched")                         '

      '--------------------------------------------------------------------------------------------+
      '- ALL VIEW                                                                                  |
      '--------------------------------------------------------------------------------------------+
      CASE "V", "VIEW"                                            '
         FOR i = 1 TO gFMDCtr                                     ' Lets look th the lines
            IF gFMD(i).Type(%FEntry, %FFLEntry) THEN              ' Only look at real file entries
               me.DoVIEW(i)                                       ' Go do View
            END IF                                                '
         NEXT i                                                   '

      CASE ELSE                                                   '
         MErrExit(%eFail, "Unsupported ALL command type: " + lcmd)   '
   END SELECT                                                     '
   '-----------------------------------------------------------------------------------------------+
   '- See if any Deferred commands                                                                 |
   '-----------------------------------------------------------------------------------------------+
   IF DefCtr > 0 THEN                                             ' Anything to do?
      ARRAY SORT Deferred(1) FOR DefCtr, DESCEND                  ' Reverse sort
      RESET t, tt                                                 ' Clear
      FOR i = 1 TO DefCtr                                         ' Elim dup tab #s (Medit files)
         IF LEFT$(Deferred(i), 3) <> t THEN                       ' New tab # ?
            t = LEFT$(Deferred(i), 3)                             ' Save it
         ELSE                                                     ' Same tab then
            MID$(Deferred(i), 4, 1) = "X"                         ' Mark it to be skipped
         END IF                                                   '
      NEXT i                                                      '
      FOR i = 1 TO DefCtr                                         ' Do the deferred commands
         IF MID$(Deferred(i), 4, 1) = "X" THEN ITERATE FOR        ' Told to skip it, then do so
         t    = LEFT$(Deferred(i), 3)                             ' Get the tab number
         tt   = MID$(Deferred(i), 5)                              ' Get the command to perform
         IF tt = "END"    THEN DoCmdInOtherTab(VAL(t), tt)        ' Do the END command
         IF tt = "CANCEL" THEN DoCmdInOtherTab(VAL(t), tt)        ' Do the CAN command
      NEXT i                                                      '
   END IF                                                         '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdBottom(pCmd AS STRING)                              '
'--------------------------------------------------------------------------------------------------+
'- BOTTOM in FM Mode                                                                               |
'--------------------------------------------------------------------------------------------------+
   MEntry                                                         '
   LastTop = 0                                                    ' Allow movement
   @P.PTopLine = MAX(1, MIN(gFMDCtr, 9999999) - (gENV.ScrHeight + 1 - (3 * gENV.FMHelpFlag) - gFM_Top_File_Line) + 1)   ' Move TopScrn
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdBrowse(pCmd AS STRING)                              '
'--------------------------------------------------------------------------------------------------+
'- BROWSE in FM Mode                                                                               |
'--------------------------------------------------------------------------------------------------+
LOCAL t AS STRING                                                 '
   t = GetNextWord(pCmd, %Strip)                                  ' Strip off command
   me.CallTab("BROWSE", pCmd)                                     ' Do a normal Browse
END METHOD                                                        '

METHOD  FMPCmdCD(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- CD in FM Mode                                                                                   |
'--------------------------------------------------------------------------------------------------+
LOCAL t, tt, NewFolder AS STRING                                  '
   IF FMode <> %FMFilePath THEN                                   ' Kill if not FilePath mode
      TP.ErrMsgAdd(%eFail, "CD only supported in FM FilePath Mode")  '
      EXIT METHOD                                                 '
   END IF                                                         '
   tt = PCmd                                                      '
   t = GetNextWord(tt, %Strip)                                    ' Remove the CD
   t = GetNextWord(tt, %NOStrip)                                  ' Get the operand

   '-----------------------------------------------------------------------------------------------+
   '- Handle quoted or unquoted versions                                                           |
   '-----------------------------------------------------------------------------------------------+
   IF INSTR($DQ + "'`", LEFT$(t, 1)) <> 0 AND _                   ' IF a leading " ' OR `
      LEFT$(t, 1) = RIGHT$(t, 1) THEN                             ' and trailing = leading
      t = MID$(t, 2, LEN(t) - 2)                                  ' Remove them
   END IF                                                         '
   IF MID$(t, 2, 1) = ":" THEN                                    ' Fully qualified?
      NewFolder = t                                               '
   ELSE                                                           '
      NewFolder = FPath + t                                       ' Change to lower folder
   END IF                                                         '

   '-----------------------------------------------------------------------------------------------+
   '- Make sure it exists                                                                          |
   '-----------------------------------------------------------------------------------------------+
   IF ISFALSE ISFOLDER(NewFolder) THEN                            ' Better exist
      TP.ErrMsgAdd(%eFail, "Specified folder does not exist")     ' No? Kill it
      EXIT METHOD                                                 '
   END IF                                                         '
   FPath = NewFolder                                              '
   FileListNm = ""                                                ' Kill FileList name
   DoSet(%LoadReq)                                                ' Full refresh
END METHOD                                                        '

METHOD  FMpCmdCSV(pCmd AS STRING)                                 '
'--------------------------------------------------------------------------------------------------+
'- Create CSV file of the FM data                                                                  |
'--------------------------------------------------------------------------------------------------+
LOCAL t1, t2, cField AS STRING, fattr AS WSTRING                  '
LOCAL i, j, k, fcol, ALIGN AS LONG                                '
LOCAL recs AS LONG                                                '
LOCAL CB AS ISTRINGBUILDERA                                       '

   MEntry                                                         '
   IF PARSECOUNT(pCmd, " ") > 1 THEN                              ' An operand
      IF UCASE$(PARSE$(pCmd, " ", 2)) <> "ALIGN" THEN MErrExit(%eFail, "Unknown operand") '
      ALIGN = %TRUE                                               ' Set align option
   END IF                                                         '
   DIM cWidth(1 TO gFMCCtr) AS LONG                               '
   CB = CLASS "StringbuilderA"                                    ' Define CB
   CB.Clear                                                       ' Clear stringbuilder, just in case
   CB.Capacity = 65536                                            ' Set a starting capacity

   gKeyPrimOper = ""                                              ' Set clipboardname
   ErrMsgHigh = %eNone                                            ' Reset any Pending conflict flag
   CB.Clear                                                       ' Start as ""

   '-----------------------------------------------------------------------------------------------+
   '- If ALIGN calc column widths                                                                  |
   '-----------------------------------------------------------------------------------------------+
   IF ALIGN THEN                                                  ' Need to calc column widths?
      FOR i = 1 TO gFMDCtr                                        ' Lets look at the lines
         IF gFMD(i).Type(%FEntry, %FFLEntry) THEN                 ' Only look at real file entries
            k = LEN(TRIM$(gFMD(i).FullPath)) + 2                  ' Get length of Name field
            cWidth(1) = MAX(k, cWidth(1))                         ' Save high-water
            '--------------------------------------------------------------------------------------+
            '- Now do the optional entries                   '                                     |
            '--------------------------------------------------------------------------------------+
            FOR j = 2 TO gFMCCtr                                  ' Do Active Columns
               cField = gFMC(j).CField                            ' Get fieldname
               t2 = gFMD(i).PrtField(cField, fAttr, fCol)         ' Then it's actual data
               IF VERIFY(t2, " 0123456789,") = 0 THEN             ' A Numeric field?
                  t2 = REMOVE$(t2, ",")                           ' Remove commas from numeric fields
                  cWidth(j) = MAX(LEN(TRIM$(t2)), cWidth(j))      ' Save high-water, no quotes
               ELSE                                               '
                  cWidth(j) = MAX(LEN(TRIM$(t2)) + 2, cWidth(j))  ' Save high-water, with quotes
               END IF                                             '
            NEXT i                                                '
         END IF                                                   '
      NEXT i                                                      '
   END IF                                                         '

   '-----------------------------------------------------------------------------------------------+
   '- Create CSV column heading line                                                               |
   '-----------------------------------------------------------------------------------------------+
   t1 = ""                                                        ' Start a null line
   FOR i = 1 TO gFMCCtr                                           ' Loop through columns
      t2 = RTRIM$(gFMC(i).CHead, ANY "+-*")                       ' Get raw name, remove +-*
      cwidth(i) = MAX(cwidth(i), LEN(t2))                         ' In case data is shorter than the header
      IF ALIGN THEN t2 = LSET$(t2, cwidth(i))                     ' Pad if ALIGN
      t1 += T2 + ","                                              ' Add the column header
   NEXT i                                                         '
   t1 = CLIP$(RIGHT, t1, 1)                                       ' Remove trailing comma

   CB.Add(t1 + $CRLF)                                             ' Add the header line
   INCR recs                                                      ' Count lines for message

   '-----------------------------------------------------------------------------------------------+
   '- Add the data lines                                                                           |
   '-----------------------------------------------------------------------------------------------+
   FOR i = 1 TO gFMDCtr                                           ' Lets look at the lines
      T1 = ""                                                     ' Create a null line
      IF gFMD(i).Type(%FEntry, %FFLEntry) THEN                    ' Only look at real file entries
         T2 = $DQ + TRIM$(gFMD(i).FullPath) + $DQ                 ' Add the filename entry
         IF ALIGN THEN t2 = LSET$(t2, cWidth(1))                  ' Padd if ALIGN
         t1 += T2 + ","                                           ' Add the comma
         '-----------------------------------------------------------------------------------------+
         '- Now add optional entries                                                               |
         '-----------------------------------------------------------------------------------------+
         FOR j = 2 TO gFMCCtr                                     ' Do Active Columns
            cField = gFMC(j).CField                               ' Get fieldname
            t2 = TRIM$(gFMD(i).PrtField(cField, fAttr, fCol))     ' Then it's actual data
            IF VERIFY(t2," 0123456789,") = 0 THEN                 ' If numeric field
               t2 = REMOVE$(t2, ",")                              ' Remove commas from numeric fields
               IF ALIGN THEN t2 = RSET$(t2, cWidth(j))            ' Padd if ALIGN
            ELSE                                                  ' Non-numeric
               t2 = $DQ + TRIM$(t2) + $DQ                         ' Add the field
               IF ALIGN THEN t2 = LSET$(t2, cWidth(j))            ' Padd if ALIGN
            END IF                                                '
            t1 += T2 + ","                                        ' Add the comma
         NEXT i                                                   '
         t1 = CLIP$(RIGHT, t1, 1)                                 ' Remove trailing comma

         '-----------------------------------------------------------------------------------------+
         '- Now write the line                                                                     |
         '-----------------------------------------------------------------------------------------+
         CB.Add(t1 + $CRLF)                                       ' Add the filename entry
         INCR recs                                                ' Count records
      END IF                                                      '
   NEXT i                                                         '

   IF ISTRUE ClipboardWrite(CB.String) THEN                       ' OK?
      TP.ErrMsgAdd(0, FORMAT$(recs) + IIF$(recs <> 1, " lines", " line") + " written to the Clipboard")  '
   ELSE                                                           '
      TP.ErrMsgAdd(%eFail, "CSV creation failed")                 '
   END IF                                                         '
   RESET gKeyPrimOper                                             ' Blank clipboardname
   MExit                                                          '
END METHOD                                                        '

METHOD  FMpCmdCUT(pCmd AS STRING)                                 '
'--------------------------------------------------------------------------------------------------+
'- CUT filenames to the Windows Clipboard                                                          |
'--------------------------------------------------------------------------------------------------+
LOCAL CBD, cbname, MyDLM, t, u AS STRING                          '
LOCAL i AS LONG                                                   '
LOCAL AppendText, recs, Raw, NewOnly AS LONG                      '
LOCAL CB AS ISTRINGBUILDERA                                       '

   MEntry                                                         '
   u = GetNextWord(pCmd, %Strip) + " "                            ' Remove 1st word (the command)
   t = "FMCUT " + pCmd                                            ' Change name to FM version
   pCmd = u + pCmd                                                ' Put back the orig pCmd version
   '-----------------------------------------------------------------------------------------------+
   '- Do the basic parsing stuff                                                                   |
   '-----------------------------------------------------------------------------------------------+
   Call3(PTBL.ParseCmd(t, %PAll), _                               ' Try the Parse now
         PCmdHelp("H CUT"), _                                     ' ? entered
         MErrExit(%eFail, PTBL.ErrMsg), _                         ' Error, Bail out
         Nul)                                                     ' Continue
   IF gENV.CutNew THEN NewOnly = %True                            ' Get global option for NEW
   CB = CLASS "StringbuilderA"                                    ' Define CB
   CB.Clear                                                       ' Clear stringbuilder, just in case
   CB.Capacity = 65536                                            ' Set a starting capacity

   t = PTBL.Grp("NEWREP")                                         ' Get NEW / REPLACE
   IF t = "NEW" THEN NewOnly = %True                              ' Set flags
   IF t = "REPLACE" THEN NewOnly = %False                         '
   IF PTBL.IsKwd("RAW") THEN Raw = %True                          '
   IF PTBL.IsKwd("APPEND") THEN AppendText = %True                '
   IF PTBL.FlgLit1 THEN cbname = PTBL.Pos2(1)                     ' Grab any cbname

   '-----------------------------------------------------------------------------------------------+
   '- Handle NEW test                                                                              |
   '-----------------------------------------------------------------------------------------------+
   IF cbname <> "" AND ISTRUE NewOnly THEN                        ' If NEW, make sure OK
      IF ISFILE(gENV.Homedata + "CLIP\" + cbname + ".CLIP") THEN  ' Does file exist?
         i = DoMessageBox("NEW specified, but |K" + gENV.Homedata + "CLIP\" + cbname + ".CLIP" + "|B already exists." + $CRLF + _ '
                          "Do you want to over-write the existing file?", %MB_YesNo + %MB_UserIcon, "SPFLite") '
         IF i = %IdNo THEN                                        '
            MErrExit(%eFail, "CUT cancelled")                     ' OK, go no further
         END IF                                                   '
      END IF                                                      '
   END IF                                                         '

   gKeyPrimOper = cbname: MyDLM = $CRLF                           ' Set clipboardname / delimiter
   ErrMsgHigh = %eNone                                            ' Reset any Pending conflict flag
   CB.Clear                                                       ' Start as ""

   IF AppendText THEN                                             '
      ClipboardRead(CBD, MyDLM, %False)                           ' Go get whatever's there, keep the file
      CB.Add(CBD)                                                 ' Stuff into new CB
      CBD = ""                                                    ' Free CBD
   END IF                                                         '

   '-----------------------------------------------------------------------------------------------+
   '- Look for the marked C/CC lines                                                               |
   '-----------------------------------------------------------------------------------------------+
   FOR i = 1 TO gFMDCtr                                           ' Lets look at the lines
      IF gFMD(i).Type(%FEntry, %FFLEntry, %FOpen) THEN            ' Only look at real file entries
         IF gFMD(i).CmdRoot = "COPY" THEN                         ' A copy cmd?
            gFMD(i).Cmd = ""                                      ' Clear it
            INCR recs                                             ' Count records
            IF ISFALSE Raw THEN                                   ' Normal mode
               CB.Add($DQ + TRIM$(gFMD(i).FullPath) + $DQ + MyDLM)   ' Add each line with delimiter
            ELSE                                                  ' Raw mode
               CB.Add($DQ + TRIM$(gFMD(i).FullPath) + $DQ)        ' Add each line without a delimiter
            END IF                                                '
         END IF                                                   '
      END IF                                                      '
   NEXT i                                                         '

   IF ISTRUE ClipboardWrite(CB.String) THEN                       ' OK?
      IF recs > 0 THEN                                            '
         IF AppendText THEN                                       ' Issue appropriate message
            TP.ErrMsgAdd(0, FORMAT$(recs) + IIF$(recs > 1, " lines", " line") + " appended to " + IIF$(ISNULL(gKeyPrimOper), "Windows", gKeyPrimOper) + " clipboard")   '
         ELSE                                                     '
            TP.ErrMsgAdd(0, FORMAT$(recs) + IIF$(recs > 1, " lines", " line") + " cut to " + IIF$(ISNULL(gKeyPrimOper), "Windows", gKeyPrimOper) + " clipboard")  '
         END IF                                                   '
      ELSE                                                        '
         TP.ErrMsgAdd(%eFail, "Cut to clipboard failed")          '
      END IF                                                      '
   END IF                                                         '
   RESET gKeyPrimOper                                             ' Blank clipboardname
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdDOWN(pCmd AS STRING)                                '
'--------------------------------------------------------------------------------------------------+
'- DOWN in FM Mode                                                                                 |
'--------------------------------------------------------------------------------------------------+
LOCAL thispage, onepage, csrpage, j AS LONG, t, u AS STRING       '
   MEntry                                                         '
   t = TRIM$(pCmd)                                                ' Trim it
   IF PARSECOUNT(t, " ") > 1 THEN                                 ' An operand
      u = UCASE$(PARSE$(t, " ", 2))                               ' Grab it
   ELSE                                                           '
      u = FCB.ScrollAmt                                           ' Else grab default
   END IF                                                         '
   onepage = (gENV.ScrHeight + 1 - (3 * gENV.FMHelpFlag) - gFM_Top_File_Line) '
   csrpage = IIF((@P.C.CRow - gFM_Top_File_Line) < 1, onepage, (@P.C.CRow - gFM_Top_File_Line)) '

   LastTop = 0                                                    ' Allow movement
   IF VAL(u) > 0 THEN                                             ' If numeric, use it
      thispage = VAL(u)                                           '
      j = 0                                                       '
   ELSE                                                           '
      u = UUCASE(LSET$(u, 4))                                     ' pad to 4
      j = INT(INSTR("    M   MAX P   PAGEF   FULLH   HALFD   DATAC   CSR ", u) / 4) '
      IF j = 0 THEN MErrExit(%eFail,"Invalid Scroll amount")      '
      thispage = CHOOSE(j, 9999999,    9999999,      onepage,     onepage,     onepage, onepage, _ ' M/Max,  P/Page, F/Full
                           onepage / 2, onepage / 2, onepage - 1, onepage - 1, onepage, onepage)   ' H/Half, D/Data, C/Csr
   END IF                                                         '
   IF j = 1 OR j = 2 THEN                                         ' MAX
      @P.PTopLine = MAX(1, MIN(gFMDCtr, 9999999) - onepage + 1)   ' Move TopScrn
   ELSEIF j = 11 OR j = 12 THEN                                   ' CSR
      @P.PTopLine = @P.PTopLine + csrpage                         ' Adjust top
      IF @P.PTopLine > gFMDCtr THEN @P.PTopLine = MAX(1, MIN(gFMDCtr, 9999999) - onepage + 1)   '
   ELSE                                                           '
      @P.PTopLine = MIN(gFMDCtr, @P.PTopLine + thispage)          ' Move TopScrn
   END IF                                                         '
   FMMarkdSLin = 0                                                ' Clear marked line
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdEdit(pCmd AS STRING)                                '
'--------------------------------------------------------------------------------------------------+
'- EDIT in FM Mode                                                                                 |
'--------------------------------------------------------------------------------------------------+
LOCAL t AS STRING                                                 '
   t = GetNextWord(pCmd, %Strip)                                  ' Strip off command
   me.CallTab("EDIT", pCmd)                                       ' Do a normal Edit
END METHOD                                                        '

METHOD  FMPCmdEnd(pCmd AS STRING)                                 '
'--------------------------------------------------------------------------------------------------+
'- END in FM Mode                                                                                  |
'--------------------------------------------------------------------------------------------------+
LOCAL J AS LONG                                                   '
   MEntry                                                         '
   IF gfEndAll THEN                                               ' Driven by EXIT?
      TabMode = (TabMode AND (&HFFFFFFFF - %FMTab))               ' Remove FM status
      me.TabTitleSet(%True)                                       '
      pCmdEnd("END")                                              ' Call END
      MExitMeth                                                   ' All done here
   END IF                                                         '

   IF ISNOTNULL(FileListNm) THEN                                  ' In filelist mode, Go back a nest level
      IF FileListNm = "Recent Files" OR FileListNm = "Recent Paths"   OR _ ' If a go back to FilePath type
         FileListNm = "Found Files"  OR FileListNm = "Favorite Files" OR _ '
         FileListNm = "FLISTS"       OR FileListNm = "Open Files"     THEN '
         FileListNm = ""                                          ' Null the FileList
      ELSE                                                        ' Else it's a private FLIST
         FMNestBack                                               ' Go back a Nest level
      END IF                                                      '
   ELSE                                                           '
      j = INSTR(-2, FPath, "\")                                   ' Find last \
      IF j <> 0 THEN                                              ' Got one
         FPath = LEFT$(FPath, j)                                  ' Cut off the last level
         gSQL.UpdateString("O", "DefDir1", FPath)                 '
      END IF                                                      ' Else do nothing
      DoSet(%LoadReq)                                             ' Request refresh
   END IF                                                         '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdExclude(pCmd AS STRING)                             '
'--------------------------------------------------------------------------------------------------+
'- EXCLUDE in FM Mode                                                                              |
'--------------------------------------------------------------------------------------------------+
REGISTER i AS LONG                                                '
REGISTER j AS LONG                                                '
   MEntry                                                         '
   IF PARSECOUNT(pCmd, " ") < 2 THEN _                            ' An operand?
      MErrExit(%eFail, "EXCLUDE requires an operand")             ' Oops
   FSearch = UCASE$(PARSE$(pCmd, " ", 2))                         ' Grab the operand
   StrUnquote(FSearch)                                            ' Remove any quotes
   LastTop = 0                                                    ' Allow movement
   FOR i = 1 TO gFMDCtr                                           ' Lets do a search for matches
      IF gFMD(i).Flag = %FEntry OR gFMD(i).Flag = %FFLEntry OR gFMD(i).Flag = %FFileList THEN   '
         IF ISTRUE me.TestMask(TRIM$(gFMD(i).FileName), FSearch) THEN   ' If we pass the Mask test
            me.IncRXSize                                          ' Allocate an RX slot
            gFMRXList(gFMRXCount) = TRIM$(gFMD(i).DPath) + TRIM$(gFMD(i).FileName)  ' Create an Exclude entry
            DoSet(%LoadData)                                      ' Have it take effect
            INCR j                                                ' Count items
         END IF                                                   '
      END IF                                                      '
   NEXT i                                                         '
   DoSet(%LoadData)                                               '
   TP.ErrMsgAdd(%eNone, IIF$(j = 0, "No entries excluded", FORMAT$(j) + " entries excluded"))   '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMpCmdFC(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- FChange - Execute a Change command against an FM list of files                                  |
'--------------------------------------------------------------------------------------------------+
LOCAL fcmd, t, X, pDiag AS STRING, RCA AS RCArea                  '
LOCAL i, j, V, lclNF, FNum, ChgCnt, ONCE, AllCount1, AllCount2 AS LONG
LOCAL OldFile, cCmd AS STRING                                     '
DIM foundlist(1 TO 100) AS STRING                                 '

   MEntry
   cCmd = UCASE$(pCmd) + " "
   cCmd = IIF$(INSTR(cCmd, " ALL "), pCmd, pCmd + "ALL ")         ' Ensure ALL is coded
   i =DoMessageBox("|KFCHANGE|B is a powerful command and a simple TYPO" + $CRLF + _
                   "can really mess up the file. You have entered:" + $CRLF + _
                   "   |K" + pCmd + "|B" + $CRLF + _
                   "Please review and select |KContinue|B or |KExit" + $CRLF + _
                   "Continue" + $CRLF + _
                   "Exit", %MB_CUSTOM2 + %MB_UserIcon + %MB_DefButton1, "SPFLite") '
         IF i = %IDCustom2 THEN MExitMeth                         ' Bail out if requested

   '-----------------------------------------------------------------------------------------------+
   '- Spin through the displayed files                                                             |
   '-----------------------------------------------------------------------------------------------+
   gfInterrupt = %False                                           '
   DispInterrupt("Click to Interrupt the Change", "Processing file: ")   ' Display the Interrupt popup

   FOR i = 1 TO gFMDCtr                                           ' Lets see if any lines were selected
      IF gfInterrupt THEN EXIT FOR                                '
      IF gFMD(i).Type(%FEntry, %FFLEntry) THEN                    ' Existing File?
         OldFile = gFMD(i).FullPath                               ' Get a file name

         CONTROL SET TEXT ghIntr, %INTERRUPT_TEXT2, RTRIM$(MID$(OldFile, INSTR(-1, OldFile, "\") + 1))
         INCR AllCount1                                           '
         CONTROL SET TEXT ghIntr, %INTERRUPT_TEXT3, "Processed: " + FORMAT$(AllCount1)   '
         DIALOG DOEVENTS 0                                        ' Let others process
         RESET gLoopCtr                                           ' Reset LoopCtr to avoid treating as a loop
         me.LInitTxtData()                                        ' Initialize our Text area
         FCB.SetupFN(OldFile, %ProfGetY, RCA)                     ' Setup filename
         IF RCA.RC <> 0 THEN ITERATE FOR                          ' Profile failed, skip
         IF me.InitaFile(%True) THEN ITERATE FOR                  ' Initialize file stuff, honor SKIP
         me.RangeSetRange(2, LastLine - 1)                        ' Everything for search
         '-----------------------------------------------------------------------------------------+
         '- Now do the change command                                                              |
         '-----------------------------------------------------------------------------------------+

         ONCE = %False                                            ' Reset
         '-----------------------------------------------------------------------------------------+
         '- Setup the Change command model                                                         |
         '-----------------------------------------------------------------------------------------+
         Call3(PTBL.ParseCmd(cCmd, %PAll), _                      ' Try the basic parse
              pCmdHelp("HELP FChange"), _                         ' ? entered
              GOTO ErrorOut, _                                    ' Error, Bail out
              Nul)                                                ' Continue

         j = me.Search(0, %CCmd)                                  ' Do the initial search
         IF j THEN                                                ' See if we have one
            me.Change()                                           ' Go do the Change
            ONCE = %True                                          ' Remember we changed something
         END IF                                                   '
         DO WHILE j                                               ' Do till not found
            j = me.Search(1, %CNext)                              ' Do re-Finds
            IF j THEN                                             '
               me.Change()                                        '
               ONCE = %True                                       ' Remember we changed something
            END IF                                                '
         LOOP                                                     '
         '-----------------------------------------------------------------------------------------+
         '- Now save as the new version if changed                                                 |
         '-----------------------------------------------------------------------------------------+
         IF ONCE THEN                                             ' If changed
            INCR AllCount2                                        ' Count
            pCmdSAVE("SAVE ")                                     ' Call SAVE
            gFMD(i).Cmd = "=CHG>"                                 '
            DoSet(%Msg)                                           ' Get msg issued
         END IF                                                   '
         me.CleanTabData                                          ' Shut Edit down
      END IF                                                      '
   NEXT i                                                         '
   TP.ErrMsgReset                                                 ' Clear any errors from Edit
   TabMode = %FMTab                                               ' Ensure we're back in FM Mode
   ErrMsgHigh = 0                                                 '
   DIALOG END ghIntr                                              ' End the window
   FCB.SetupProf("DEFAULT", RCA)                                  ' Overlay with DEFAULT if available
   gfInterrupt = %False                                           '
   LastLine = 3                                                   ' Prevent Editor opens (if last file was empty or Skipped outouf)
   MErrExit(%eNone, FORMAT$(AllCount2) + " out of " + FORMAT$(AllCount1) + " files in the display were altered")

   Errorout:
      DIALOG END ghIntr                                           ' End the window
      FCB.SetupProf("DEFAULT", RCA)                               ' Overlay with DEFAULT if available
      gfInterrupt = %False                                        '
      LastLine = 3                                                ' Prevent Editor opens (if last file was empty or Skipped outouf)
      MErrExit(%eFail, PTBL.ErrMsg)                               '
END METHOD                                                        '

METHOD  FMpCmdFF(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- FF (File in Find) command                                                                       |
'--------------------------------------------------------------------------------------------------+
LOCAL fcmd, t, X, pDiag AS STRING, RCA AS RCArea                  '
LOCAL i, j, AllCount, lclNF, FNum AS LONG, ff AS QUAD             '
DIM foundlist(1 TO 100) AS STRING                                 '
   MEntry                                                         '
   '-----------------------------------------------------------------------------------------------+
   '- Do the basic parsing stuff.                                                                  |
   '-----------------------------------------------------------------------------------------------+
   Call3(PTBL.ParseCmd(pCmd, %PAll), _                            ' Try the basic parse
         pCmdHelp("H FF"), _                                      ' ? entered
         MErrExit(%eFail, PTBL.ErrMsg), _                         ' Error, Bail out
         Nul)                                                     ' Continue
   IF PTBL.OpsNum = 0 THEN MErrExit(%eFail, "Missing FF operands")   ' Exit if nothing to do
   fcmd = pCommand                                                ' Save the whole FF command
   IF ISFALSE PTBL.FlgLit1 THEN MErrExit(%eFail, "No search literal specified")  ' Better have a Lit
   IF PTBL.FlgNF THEN lclNF = %True: PTBL.FlgNF = %False          ' If NF, save flag and then clear it

   AllCount = 0                                                   ' Reset counter
   gfInterrupt = %False                                           '
   DispInterrupt("Click to Interrupt search", "Processing file: ")   ' Display the Interrupt popup

   FOR i = 1 TO gFMDCtr                                           ' Lets see if any lines were selected
      IF gfInterrupt THEN EXIT FOR                                '
      IF gFMD(i).Type(%FEntry, %FFLEntry, %FOpen) THEN            ' Existing File?
         x = UUCASE(MID$(RTRIM$(gFMD(i).FileName), INSTR(-1, RTRIM$(gFMD(i).FileName), ".") + 1))  '
         IF GetProfileForFile(x, pDiag, %False, %True) = "NONTEXT" THEN ITERATE FOR ' (NoEFT,Query)
         CONTROL SET TEXT ghIntr, %INTERRUPT_TEXT2, RTRIM$(gFMD(i).FileName)  '
         CONTROL SET TEXT ghIntr, %INTERRUPT_TEXT3, "Found: " + FORMAT$(AllCount)   '
         DIALOG DOEVENTS 0                                        ' Let others process
         RESET gLoopCtr                                           ' Reset LoopCtr to avoid treating as a loop
         me.LInitTxtData()                                        ' Initialize our Text area
         t =gFMD(i).FullPath                                      ' Setup filename
         FCB.SetupFN(t, %ProfGetY, RCA)                           ' Setup filename
         IF RCA.RC <> 0 THEN ITERATE FOR                          ' Profile failed, skip
         IF me.InitaFile(%True) THEN ITERATE FOR                  ' Initialize file stuff, honor SKIP
         me.RangeSetRange(2, LastLine - 1)                        ' Everything for search
         j = me.Search(0, %CBad)                                  ' Normal version
         IF ISTRUE j AND ISFALSE lclNF OR _                       ' See if we have one
            ISFALSE j AND ISTRUE lclNF THEN                       '
            INCR AllCount                                         ' Count a found file
            IF AllCount > UBOUND(foundlist()) THEN _              ' Table still big enough?
               REDIM PRESERVE foundlist(1 TO 2 * UBOUND(foundlist()))   ' No, enlarge it
            foundlist(AllCount) = gFMD(i).FullPath                '
         END IF                                                   '
      END IF                                                      '
   NEXT i                                                         '
   TP.ErrMsgReset                                                 ' Clear any errors from Edit
   TabMode = %FMTab                                               ' Ensure we're back in FM Mode
   ErrMsgHigh = 0                                                 '
   DIALOG END ghIntr                                              ' End the window
   FCB.SetupProf("DEFAULT", RCA)                                  ' Overlay with DEFAULT if available
   gfInterrupt = %False                                           '
   LastLine = 3                                                   ' Prevent Editor opens (if last file was empty or Skipped outouf)

   IF AllCount = 0 THEN                                           ' Issue approp. message
      MErrExit(%eFail, "No Files match search")                   '

   ELSE                                                           '
      TP.ErrMsgAdd( %eNone, FORMAT$(AllCount) + IIF$(AllCount > 1, " files", " file") + " satisfied search.")  '
      FNum = FREEFILE                                             ' Get a file number
      Call3(TryOpenOutPut(gENV.HomeData + "FILELIST\" + "Found Files.FLIST", FNum), _   ' Try the open
            MErrExit(%eFail, "FILELIST Write Open failed"), _     ' Oops?  Bail out
            MErrExit(%eFail, "FILELIST File is in use"), _        ' Oops?  Bail out
            Nul)                                                  ' Continue
      PRINT # FNum, "FileSrch: " + fCmd                           ' Save the search string
      FOR i = 1 TO AllCount                                       ' Write the array back out
         PRINT # FNum, foundlist(i)                               '
      NEXT i                                                      '
      CLOSE #fnUM                                                 ' Close the FBO
      FileListNm = "Found Files"                                  ' Set FOUND as the new filelist
      DoSet(%LoadReq OR %Refresh)                                 ' Refresh
   END IF                                                         '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdFind(pCmd AS STRING)                                '
'--------------------------------------------------------------------------------------------------+
'- FIND in FM Mode                                                                                 |
'--------------------------------------------------------------------------------------------------+
REGISTER i AS LONG                                                '
LOCAL t, u AS STRING                                              '
   MEntry                                                         '
   u = GetNextWord(pCmd, %Strip) + " "                            ' Remove 1st word (the command)
   t = "FMFIND " + pCmd                                           ' Change name to FM version
   pCmd = u + pCmd                                                ' Put back the orig pCmd version
   '-----------------------------------------------------------------------------------------------+
   '- Do the basic parsing stuff                                                                   |
   '-----------------------------------------------------------------------------------------------+
   Call3(PTBL.ParseCmd(t, %PAll), _                               ' Try the Parse now
         PCmdHelp("H -IXFP"), _                                   ' ? entered
         MErrExit(%eFail, PTBL.ErrMsg), _                         ' Error, Bail out
         Nul)                                                     ' Continue
   t = PTBL.Grp("MODIFIER")                                       ' Get WRD/CHAR
   IF t = "WORD" THEN                                             ' Single operand Kwd WORD?
      FindWord = %True                                            '
      MErrExit(%eNone, "FIND mode set temporarily to: WORD")      '
   END IF                                                         '
   IF t = "CHAR" THEN                                             ' Single operand Kwd CHAR?
      FindWord = %True                                            '
      MErrExit(%eNone, "FIND mode set temporarily to: CHAR")      '
   END IF                                                         '
   IF PTBL.GotLit = 0 THEN MErrExit(%eFail, "Missing FIND operand")  '

   FSearch = UUCASE(PTBL.Pos2(1)): LastLoc = @P.PTopLine          ' Get search operand
   StrUnQuote(FSearch)                                            ' Remove quotes
   LastTop = 0                                                    ' Allow movement
   FOR i = LastLoc TO gFMDCtr                                     ' Lets see where it is
      IF INSTR(UUCASE(gFMD(i).FileName), FSearch) > 0 THEN        ' Found it?
         @P.PTopLine = i: LastLoc = i                             ' Set top of screen here
         me.CsrRow = gFM_Top_File_Line: me.CsrCol = @P.PLeft: MExitMeth  '
      END IF                                                      '
   NEXT i                                                         '
   TP.ErrMsgAdd(%eNone, "Bottom of list reached")                 '
   LastLoc = 1                                                    '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdLEFT(pCmd AS STRING)                                '
'--------------------------------------------------------------------------------------------------+
'- Left in FM Mode                                                                                 |
'--------------------------------------------------------------------------------------------------+
   MEntry                                                         '
   TP.DefFirstCol = MAX(TP.DefFirstCol - 1, 2)                    ' Shift 1 column left
   InitFMLayout                                                   ' Rebuild layout
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdLocate(pCmd AS STRING)                              '
'--------------------------------------------------------------------------------------------------+
'- LOCATE in FM Mode                                                                               |
'--------------------------------------------------------------------------------------------------+
LOCAL i, j AS LONG, s, l AS WSTRINGZ * %MAX_PATH, t, u AS STRING  '
   MEntry                                                         '
   u = GetNextWord(pCmd, %Strip) + " "                            ' Remove 1st word (the command)
   t = "FMLOC " + pCmd                                            ' Change name to FM version
   pCmd = u + pCmd                                                ' Put back the orig pCmd version
   '-----------------------------------------------------------------------------------------------+
   '- Do the basic parsing stuff                                                                   |
   '-----------------------------------------------------------------------------------------------+
   Call3(PTBL.ParseCmd(t, %PAll), _                               ' Try the Parse now
         PCmdHelp("H -IXFP"), _                                   ' ? entered
         MErrExit(%eFail, PTBL.ErrMsg), _                         ' Error, Bail out
         Nul)                                                     ' Continue
   IF PTBL.OpsNum = 1 AND PTBL.GotNum > 0 THEN                    ' If a numeric operand
      LastTop = 0                                                 ' Allow movement
      i = VAL(PTBL.Pos("#", 1))                                   ' Get the number
      i = MIN(i, gFMDCtr - 1)                                     ' Impose limits
      i = MAX(i, 1)                                               '
      @P.PTopLine = i: LastLoc = i                                ' Set top of screen here
      MExitMeth                                                   '
   END IF                                                         '
   IF LEFT$(TP.DefSort, 4) <> "Name" THEN _                       ' Only if sorted correctly
      MErrExit(%eFail,"LOCATE requires files in Name+ or Name- order")  '
   IF PTBL.OpsNum = 1 AND PTBL.GotLit = 0 THEN MErrExit(%eFail,"LOCATE requires a search filename")   '
   s = UUCASE(PTBL.Pos2(1))                                       ' Ucase the arg
   LastTop = 0                                                    ' Allow movement
   IF MID$(TP.DefSort, 5, 1) = "+" THEN                           ' Ascending sort?
      FOR i = 1 TO gFMDCtr                                        ' Lets see where it is
         IF gFMD(i).Type(%FEntry, %FFLEntry, %FConfig) THEN       ' If a normal file entry
            l = UUCASE(LEFT$(gFMD(i).FileName, LEN(s)))           ' Get filename into WSTRINGZ
            j = StrCmpLogicalW(s, l)                              ' Do logical compare
            IF j <= 0 THEN                                        ' Found oour place
               @P.PTopLine = MAX(i - 1, 1): LastLoc = MAX(i - 1, 1)  ' Set top of screen here
               EXIT FOR                                           '
            END IF                                                '
         END IF                                                   '
      NEXT i                                                      '
   ELSE                                                           ' No, Descending
      FOR i = 1 TO gFMDCtr                                        ' Lets see where it is
         IF gFMD(i).Type(%FEntry, %FFLEntry, %FConfig) THEN       ' If a normal file entry
            l = UUCASE(LEFT$(gFMD(i).FileName, LEN(s)))           ' Get filename into WSTRINGZ
            j = StrCmpLogicalW(s, l)                              ' Do logical compare
            IF j >= 0 THEN                                        ' Found oour place
               @P.PTopLine = MAX(i - 1, 1): LastLoc = MAX(i - 1, 1)  ' Set top of screen here
               EXIT FOR                                           '
            END IF                                                '
         END IF                                                   '
      NEXT i                                                      '
   END IF                                                         '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdMD(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- MD in FM Mode                                                                                   |
'--------------------------------------------------------------------------------------------------+
LOCAL t, tt, NewFolder AS STRING                                  '
   IF FMode <> %FMFilePath THEN                                   ' Kill if not FilePath mode
      TP.ErrMsgAdd(%eFail, "MD only supported in FM FilePath Mode")  '
      EXIT METHOD                                                 '
   END IF                                                         '
   tt = PCmd                                                      '
   t = GetNextWord(tt, %Strip)                                    ' Remove the MD
   t = GetNextWord(tt, %NOStrip)                                  ' Get the operand

   '-----------------------------------------------------------------------------------------------+
   '- Handle quoted or unquoted versions                                                           |
   '-----------------------------------------------------------------------------------------------+
   IF INSTR($DQ + "'`", LEFT$(t, 1)) <> 0 AND _                   ' IF a leading " ' OR `
      LEFT$(t, 1) = RIGHT$(t, 1) THEN                             ' and trailing = leading
      t = MID$(t, 2, LEN(t) - 2)                                  ' Remove them
   END IF                                                         '
   IF MID$(t, 2, 1) = ":" THEN                                    ' If fully qualified
      TP.ErrMsgAdd(%eFail, "MD can only create folders within the current folder")  ' Yes, Kill it
      EXIT METHOD                                                 '
   ELSE                                                           '
      NewFolder = FPath + t                                       ' Change to lower folder
   END IF                                                         '

   '-----------------------------------------------------------------------------------------------+
   '- See if it already exists                                                                     |
   '-----------------------------------------------------------------------------------------------+
   IF ISFOLDER(NewFolder) THEN                                    ' Better not exist
      TP.ErrMsgAdd(%eFail, "Specified folder already exists")     ' No? Kill it
      EXIT METHOD                                                 '
   END IF                                                         '

   '-----------------------------------------------------------------------------------------------+
   '- Try creating it                                                                              |
   '-----------------------------------------------------------------------------------------------+
   TRY                                                            '
      MKDIR NewFolder                                             '
   CATCH                                                          '
      TP.ErrMsgAdd(%eFail, "Error attempting folder creation")    ' Error? Kill it
      EXIT METHOD                                                 '
   END TRY                                                        '
   DoSet(%LoadReq)                                                ' Full refresh
END METHOD                                                        '

METHOD FMPCmdRecall(pCmd AS STRING)                               '
'--------------------------------------------------------------------------------------------------+
'- RECALL in FM Mode                                                                               |
'--------------------------------------------------------------------------------------------------+
LOCAL t AS STRING                                                 '
   MEntry                                                         '
   '-----------------------------------------------------------------------------------------------+
   '- Do the basic parsing stuff.                                                                  |
   '-----------------------------------------------------------------------------------------------+
   Call3(PTBL.ParseCmd(pCmd, %PAll), _                            ' Try the basic parse
         pCmdHelp("H RECALL"), _                                  ' ? entered
         MErrExit(%eFail, PTBL.ErrMsg), _                         ' Error, Bail out
         Nul)                                                     ' Continue
   IF PTBL.OpsNum = 0 THEN t = "Recent Files": GOTO DoSwitch      ' No ops, use RECENT
   t = PTBL.Grp("WHICH")                                          ' Nothing yet? Try the Kwds
   IF ISNOTNULL(t) THEN                                           ' A Kwd, translate it
      SELECT CASE AS CONST$ t                                     ' Replace with full name
         CASE "RECENT"      : t = "Recent Files"                  '
         CASE "FAVORITE"    : t = "Favorite Files"                '
         CASE "FOUND"       : t = "Found Files"                   '
         CASE "PATHS"       : t = "Recent Paths"                  '
         CASE "FLISTS"      : t = "FLISTS"                        '
         CASE "OPEN"        : t = "Open Files"                    '
         CASE "CONFIGS"     : t = "Configs"                       '
         CASE "FILEPATH"    : t = ""                              '
         CASE "*"           : t = "ClipBoard"                     '
      END SELECT                                                  '
      GOTO DoSwitch                                               ' Go do it
   END IF                                                         '
   IF PTBL.OpsNum > 0 AND PTBL.GotLit > 0 THEN                    ' A different name entered?
      t = PTBL.Pos2(1)                                            ' Get it
      IF ISFALSE ISFILE(gENV.HomeData + "FILELIST\" + t + ".FLIST") THEN _  ' See if the file exists
         MErrExit(%eFail,"Recall failed - " + t + ".FLIST does not exist") '
      GOTO DoSwitch                                               '
   END IF                                                         '
   MErrExit(%eFail, "No FLIST name entered")                      '
   DoSwitch:                                                      '
   FileListNm = t                                                 ' Swap to FileListNm
   DoSet(%LoadReq)                                                ' Call for a LoadReq
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdReset(pCmd AS STRING)                               '
'--------------------------------------------------------------------------------------------------+
'- RESET in FM Mode. Just does the X reset                                                                                |
'--------------------------------------------------------------------------------------------------+
   MEntry                                                         '
   gFMRXCount = 0: RESET gFMRXList()                              ' Wipe it out
   TP.ErrMsgAdd(%eNone, "RESET complete")                         '
   DoSet(%SaveReq)                                                ' Request save of this now
   DoSet(%LoadReq)                                                ' Refresh all
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdRIGHT(pCmd AS STRING)                               '
'--------------------------------------------------------------------------------------------------+
'- Right in FM Mode                                                                                |
'--------------------------------------------------------------------------------------------------+
   MEntry                                                         '
   TP.DefFirstCol = MIN(TP.DefFirstCol + 1, gFMCCtr)              ' Shift 1 column right
   InitFMLayout                                                   ' Rebuild layout
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdRFind(pCmd AS STRING)                               '
'--------------------------------------------------------------------------------------------------+
'- RFIND in FM Mode                                                                                |
'--------------------------------------------------------------------------------------------------+
LOCAL i AS LONG                                                   '
   MEntry                                                         '
   IF ISNULL(FSearch) OR LastLoc = 0 THEN _                       ' Previous success
      MErrExit(%eFail,"No prior search")                          '
   LastTop = 0                                                    ' Allow movement
   FOR i = LastLoc + 1 TO gFMDCtr                                 ' Lets see where it is
      IF INSTR(UUCASE(gFMD(i).FileName), UUCASE(FSearch)) > 0 THEN   ' Found it?
         @P.PTopLine = i: LastLoc = i                             ' Set top of screen here
         me.CsrRow = gFM_Top_File_Line: me.CsrCol = @P.PLeft: MExitMeth  '
      END IF                                                      '
   NEXT i                                                         '
   TP.ErrMsgAdd(%eNone, "Bottom of list reached")                 '
   LastLoc = 1                                                    '
   MExit                                                          '
END METHOD                                                        '

METHOD  FMPCmdTop(pCmd AS STRING)                                 '
'--------------------------------------------------------------------------------------------------+
'- Top in FM Mode                                                                                  |
'--------------------------------------------------------------------------------------------------+
   LastTop = 0                                                    ' Allow movement
   @P.PTopLine = 1                                                ' Move TopScrn
END METHOD                                                        '

METHOD  FMPCmdView(pCmd AS STRING)                                '
'--------------------------------------------------------------------------------------------------+
'- VIEW in FM Mode                                                                                 |
'--------------------------------------------------------------------------------------------------+
LOCAL t AS STRING                                                 '
   t = GetNextWord(pCommand, %Strip)                              ' Strip off command
   me.CallTab("VIEW", pCommand)                                   ' Do a normal View
END METHOD                                                        '

METHOD  FMPCmdUp(pCmd AS STRING)                                  '
'--------------------------------------------------------------------------------------------------+
'- UP in FM Mode                                                                                   |
'--------------------------------------------------------------------------------------------------+
LOCAL thispage, onepage, csrpage, j AS LONG, t, u AS STRING       '
   MEntry                                                         '
   t = TRIM$(pCmd)                                                ' Trim it
   IF PARSECOUNT(t, " ") > 1 THEN                                 ' An operand
      u = UCASE$(PARSE$(t, " ", 2))                               ' Grab it
   ELSE                                                           '
      u = FCB.ScrollAmt                                           ' Else grab default
   END IF                                                         '
   onepage = (gENV.ScrHeight + 1 - (3 * gENV.FMHelpFlag) - gFM_Top_File_Line) '
   csrpage = IIF(onepage - (@P.C.CRow - gFM_Top_File_Line) < 1, onepage, onepage - (@P.C.CRow - gFM_Top_File_Line) - 1) '
   LastTop = 0                                                    ' Allow movement
   IF VAL(u) > 0 THEN                                             ' If numeric, use it
      thispage = VAL(u)                                           '
      j = 0                                                       '
   ELSE                                                           '
      u = UUCASE(LSET$(u, 4))                                     ' pad to 4
      j = INT(INSTR("    M   MAX P   PAGEF   FULLH   HALFD   DATAC   CSR ", u) / 4) '
      IF j = 0 THEN MErrExit(%eFail,"Invalid Scroll amount")      '
      thispage = CHOOSE(j, 9999999,     9999999,     onepage,     onepage,     onepage, onepage, _ ' M/Max,  P/Page, F/Full
                           onepage / 2, onepage / 2, onepage - 1, onepage - 1, csrpage, csrpage)   ' H/Half, D/Data, C/Csr
   END IF                                                         '
   @P.PTopLine = MAX(1, @P.PTopLine - thispage)                   ' Move TopScrn
   FMMarkdSLin = 0                                                ' Clear marked line
   MExit                                                          '
END METHOD                                                        '
