'--------------------------------------------------------------------------------------------------+
'- License Stuff                                                                                   |
'-                                                                                                 |
'-                                                                                                 |
'-   SPFLite is free software: you can redistribute it and/or modify                               |
'-   it under the terms of the GNU General Public License as published by                          |
'-   the Free Software Foundation, either version 3 of the License, or                             |
'-   (at your option) any later version.                                                           |
'-                                                                                                 |
'-   SPFLite is distributed in the hope that it will be useful,                                    |
'-   but WITHOUT ANY WARRANTY; without even the implied warranty of                                |
'-   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                 |
'-   GNU General Public License for more details.                                                  |
'-                                                                                                 |
'-   You should have received a copy of the GNU General Public License                             |
'-   along with SPFLite.  If not, see <https://www.gnu.org/licenses/>.                             |
'-                                                                                                 |
'--------------------------------------------------------------------------------------------------+
'- ObjKbdT.inc                                                                                     |
'--------------------------------------------------------------------------------------------------+
                                                                  '
CLASS cKbdTable                                                   '

   '-----------------------------------------------------------------------------------------------+
   '- Keyboard Table Data                                                                          |
   '-----------------------------------------------------------------------------------------------+
   INSTANCE Kbd107()         AS iKbMastEntry                      ' Master key definition table
   INSTANCE KeyCode()        AS STRING                            ' KeyCode for each Kbd107 entry
   INSTANCE Kbd()            AS iKBActive                         ' Final lookup table built from Kbd107()
   INSTANCE KMode()          AS STRING                            ' KMode for each Kbd entry
   INSTANCE KbdPFShow()      AS STRING * 32                       ' PFSHOW data strings
   INSTANCE KbdKPEnter       AS LONG                              ' Kbd Index of the Enter Key
   INSTANCE KbdCount         AS LONG                              ' Kbd number of entries
   INSTANCE KbdPFShowNum     AS LONG                              ' PFSHOW number of entries
   INSTANCE KbdPFShowMax     AS LONG                              ' Max length of PFSHOW entry

   CLASS METHOD CREATE()                                          ' Constructor - Initialize Class stuff
      DIM Kbd107(0 TO 107)   AS INSTANCE iKbMastEntry             ' Master physical key table
      DIM KeyCode(0 TO 107)  AS INSTANCE STRING                   ' Keycode for each entry
      DIM Kbd(1 TO 750)      AS INSTANCE iKBActive                ' Final Kbd mapping table
      DIM KMode(1 TO 750)    AS INSTANCE STRING                   ' Final Kbd mapping table
      DIM KbdPFShow(200)     AS INSTANCE STRING * 32              ' PFSHOW strings
      gKbdT.LoadKBTable                                           ' Go Load the tables
      gKbdT.LoadKBFinal                                           ' Finalize it

   END METHOD                                                     '

   INTERFACE iKbdTable: INHERIT IUNKNOWN                          ' Define the interface

      GProp(KbdKPEnter, LONG)                                     ' Kbd Index of the Enter Key
      GProp(KbdPFShowNum, LONG)                                   ' PFSHOW number of entries
      GProp(KbdPFShowMax, LONG)                                   ' Max length of PFSHOW entry
      PROPERTY GET KbdPFShow(ix AS LONG) AS STRING:      PROPERTY = KbdPFShow(ix):                END PROPERTY '
      PROPERTY GET KIcon(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).KIcon:             END PROPERTY '
      PROPERTY GET X(ix AS LONG) AS LONG:                PROPERTY = Kbd107(ix).X:                 END PROPERTY '
      PROPERTY GET Y(ix AS LONG) AS LONG:                PROPERTY = Kbd107(ix).Y:                 END PROPERTY '
      PROPERTY GET W(ix AS LONG) AS LONG:                PROPERTY = Kbd107(ix).W:                 END PROPERTY '
      PROPERTY GET H(ix AS LONG) AS LONG:                PROPERTY = Kbd107(ix).H:                 END PROPERTY '
      PROPERTY GET Labl(ix AS LONG) AS STRING:           PROPERTY = Kbd107(ix).Labl:              END PROPERTY '
      PROPERTY GET UpKey(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).UpKey:             END PROPERTY '
      METHOD SetUpKey(ix AS LONG, vl AS STRING):         Kbd107(ix).UpKey = vl:                   END METHOD   '
      PROPERTY GET NData(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).NData:             END PROPERTY '
      METHOD SetNData(ix AS LONG, vl AS STRING):         Kbd107(ix).NData = vl:                   END METHOD   '
      PROPERTY GET SData(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).SData:             END PROPERTY '
      METHOD SetSData(ix AS LONG, vl AS STRING):         Kbd107(ix).SData = vl:                   END METHOD   '
      PROPERTY GET CData(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).CData:             END PROPERTY '
      METHOD SetCData(ix AS LONG, vl AS STRING):         Kbd107(ix).CData = vl:                   END METHOD   '
      PROPERTY GET AData(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).AData:             END PROPERTY '
      METHOD SetAData(ix AS LONG, vl AS STRING):         Kbd107(ix).AData = vl:                   END METHOD   '
      PROPERTY GET SAData(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).SAData:            END PROPERTY '
      METHOD SetSAData(ix AS LONG, vl AS STRING):        Kbd107(ix).SAData = vl:                  END METHOD   '
      PROPERTY GET SCData(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).SCData:            END PROPERTY '
      METHOD SetSCData(ix AS LONG, vl AS STRING):        Kbd107(ix).SCData = vl:                  END METHOD   '
      PROPERTY GET CAData(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).CAData:            END PROPERTY '
      METHOD SetCAData(ix AS LONG, vl AS STRING):        Kbd107(ix).CAData = vl:                  END METHOD   '
      PROPERTY GET SCAData(ix AS LONG) AS STRING:        PROPERTY = Kbd107(ix).SCAData:           END PROPERTY '
      METHOD SetSCAData(ix AS LONG, vl AS STRING):       Kbd107(ix).SCAData = vl:                 END METHOD   '
      PROPERTY GET NAllow(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).NAllow:            END PROPERTY '
      METHOD SetNAllow(ix AS LONG, vl AS STRING):        Kbd107(ix).NAllow = vl:                  END METHOD   '
      PROPERTY GET SAllow(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).SAllow:            END PROPERTY '
      METHOD SetSAllow(ix AS LONG, vl AS STRING):        Kbd107(ix).SAllow = vl:                  END METHOD   '
      PROPERTY GET CAllow(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).CAllow:            END PROPERTY '
      METHOD SetCAllow(ix AS LONG, vl AS STRING):        Kbd107(ix).CAllow = vl:                  END METHOD   '
      PROPERTY GET AAllow(ix AS LONG) AS STRING:         PROPERTY = Kbd107(ix).AAllow:            END PROPERTY '
      METHOD SetAAllow(ix AS LONG, vl AS STRING):        Kbd107(ix).AAllow = vl:                  END METHOD   '
      PROPERTY GET NUp(ix AS LONG) AS STRING:            PROPERTY = Kbd107(ix).NUp:               END PROPERTY '
      METHOD SetNUp(ix AS LONG, vl AS STRING):           Kbd107(ix).NUp = vl:                     END METHOD   '
      PROPERTY GET SUp(ix AS LONG) AS STRING:            PROPERTY = Kbd107(ix).SUp:               END PROPERTY '
      METHOD SetSUp(ix AS LONG, vl AS STRING):           Kbd107(ix).SUp = vl:                     END METHOD   '
      PROPERTY GET CUp(ix AS LONG) AS STRING:            PROPERTY = Kbd107(ix).CUp:               END PROPERTY '
      METHOD SetCUp(ix AS LONG, vl AS STRING):           Kbd107(ix).CUp = vl:                     END METHOD   '
      PROPERTY GET AUp(ix AS LONG) AS STRING:            PROPERTY = Kbd107(ix).AUp:               END PROPERTY '
      METHOD SetAUp(ix AS LONG, vl AS STRING):           Kbd107(ix).AUp = vl:                     END METHOD   '
      PROPERTY GET SAUp(ix AS LONG) AS STRING:           PROPERTY = Kbd107(ix).SAUp:              END PROPERTY '
      METHOD SetSAUp(ix AS LONG, vl AS STRING):          Kbd107(ix).SAUp = vl:                    END METHOD   '
      PROPERTY GET SCUp(ix AS LONG) AS STRING:           PROPERTY = Kbd107(ix).SCUp:              END PROPERTY '
      METHOD SetSCUp(ix AS LONG, vl AS STRING):          Kbd107(ix).SCUp = vl:                    END METHOD   '
      PROPERTY GET CAUp(ix AS LONG) AS STRING:           PROPERTY = Kbd107(ix).CAUp:              END PROPERTY '
      METHOD SetCAUp(ix AS LONG, vl AS STRING):          Kbd107(ix).CAUp = vl:                    END METHOD   '
      PROPERTY GET SCAUp(ix AS LONG) AS STRING:          PROPERTY = Kbd107(ix).SCAUp:             END PROPERTY '
      METHOD SetSCAUp(ix AS LONG, vl AS STRING):         Kbd107(ix).SCAUp = vl:                   END METHOD   '
      PROPERTY GET ForceUp(ix AS LONG) AS STRING:        PROPERTY = Kbd107(ix).ForceUp:           END PROPERTY '
      METHOD SetForceUp(ix AS LONG, vl AS STRING):       Kbd107(ix).ForceUp = vl:                 END METHOD   '

      METHOD SupSent(ix AS LONG) AS STRING:              METHOD   = Kbd(ix).SupSent:              END METHOD   '
      METHOD SetSupSent(ix AS LONG, v AS STRING):        Kbd(ix).SupSent = v:                     END METHOD   '
      METHOD SupRepeat(ix AS LONG) AS STRING:            METHOD   = Kbd(ix).SupRepeat:            END METHOD   '
      METHOD SetSupRepeat(ix AS LONG, v AS STRING):      Kbd(ix).SupRepeat = v:                   END METHOD   '
      METHOD kData(ix AS LONG) AS STRING:                METHOD = Kbd(ix).KData:                  END METHOD   '

      METHOD SCAN(Lookup AS STRING) AS LONG                       '
      '--------------------------------------------------------------------------------------------+
      '- Look for the key/mode combo                                                               |
      '--------------------------------------------------------------------------------------------+
      REGISTER i AS LONG                                          '
         ARRAY SCAN KMode() FOR KbdCount, =Lookup, TO i           '
         METHOD = i                                               ' Return ARRAY SCAN result
      END METHOD                                                  '

      METHOD ScanFull(Lookup AS STRING) AS LONG                   '
         REGISTER i AS LONG                                       '
         LOCAL k AS STRING                                        '
         k = LEFT$(Lookup, 1)                                     ' Get 1st byte
         ARRAY SCAN KeyCode(), =k, TO i                           ' Look for KeyCode
         METHOD = i                                               ' Return Array scan result
      END METHOD                                                  '

      METHOD KBGetMastKey(kNum AS LONG, kID AS LONG, kData AS STRING, kUp AS STRING)   '
      LOCAL kbdata, tx AS STRING, kbdIX, phase, kIX, j AS LONG    '
      DIM Args(10) AS STRING                                      '

         '-----------------------------------------------------------------------------------------+
         '- Get Master table data for a single key                                                 |
         '-----------------------------------------------------------------------------------------+
         kbData = RESOURCE$(RCDATA, "KBDATA")                     ' Get kbData from our RES file default
         j = PARSECOUNT(kbData, $CRLF)                            ' Loop through the data finally
         FOR kbDix = 1 TO j                                       '
            TX = PARSE$(kbData, $CRLF, kbdIX)                     ' Extract a line

            '--------------------------------------------------------------------------------------+
            '- Comment lines                                                                       |
            '--------------------------------------------------------------------------------------+
            IF LEFT$(TX, 1) = ";" OR ISNULL(TRIM$(TX)) THEN       ' Skip comments, reset phase
               phase = 1                                          '
               ITERATE FOR                                        '
            END IF                                                '
            SELECT CASE AS LONG phase                             ' Which line of Set?

               '-----------------------------------------------------------------------------------+
               '- First line of a set                                                              |
               '-----------------------------------------------------------------------------------+
               CASE 1                                             ' First
                  PARSE TX, Args()                                ' Break out the 'words'
                  INCR kIX                                        ' Bump Table index
                  IF kIX > kNum THEN EXIT FOR                     ' We're done
                  INCR phase                                      ' Skip

               '-----------------------------------------------------------------------------------+
               '- Second line of a set (Normal Key)                                                |
               '-----------------------------------------------------------------------------------+
               CASE 2                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_NORMAL THEN     ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Third line of a set (Shift Key)                                                  |
               '-----------------------------------------------------------------------------------+
               CASE 3                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_SHIFT THEN      ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Fourth line of a set (Control Key)                                               |
               '-----------------------------------------------------------------------------------+
               CASE 4                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_CONTROL THEN    ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Fifth line of a set (ALT Key)                                                    |
               '-----------------------------------------------------------------------------------+
               CASE 5                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_ALT THEN        ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Sixth line of a set (Shift-Control Key)                                          |
               '-----------------------------------------------------------------------------------+
               CASE 6                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_SHIFTCONTROL THEN  ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Seventh line of a set (Shift-ALT Key)                                            |
               '-----------------------------------------------------------------------------------+
               CASE 7                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_SHIFTALT THEN   ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Eighth line of a set (Shift-Control-ALT Key)                                     |
               '-----------------------------------------------------------------------------------+
               CASE 8                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_SHIFTCONTALT THEN  ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Ninth line of a set (Control-ALT Key)                                            |
               '-----------------------------------------------------------------------------------+
               CASE 9                                             ' Normal command
                  IF kNum = kIX AND kID = %KEYMAP_CONTROLALT THEN ' Desired one?
                     kData = MID$(TX, 3)                          ' Save key data
                     kUp = LEFT$(TX, 1)                           '
                  END IF                                          '
                  INCR phase                                      ' Next phase
            END SELECT                                            '
         NEXT kbDIX                                               ' Loop through kbData
      END METHOD                                                  '

      METHOD LoadKbTable()                                        '
      LOCAL kIX, phase, kbDIX, i, j, k AS LONG, TX, kbData AS STRING '
      DIM args(10) AS STRING                                      '
         '-----------------------------------------------------------------------------------------+
         '- Build dummy entry as Key 0 (Used as a PASSTHRU for extra keys on foreign keyboards)    |
         '-----------------------------------------------------------------------------------------+
         IF ISNOTHING(Kbd107(0)) THEN LET Kbd107(0) = CLASS "cKBMastEntry" ' Build a dummy key entry
         KeyCode(0) = CHR$(0)                                     '
         Kbd107(0).Extern = " "                                   '
         Kbd107(0).KIcon = " "                                    '
         Kbd107(0).W = 0                                          '
         Kbd107(0).H = 0                                          '
         Kbd107(0).X = 0                                          '
         Kbd107(0).Y = 0                                          '
         Kbd107(0).NAllow = "N"                                   '
         Kbd107(0).SAllow = "S"                                   '
         Kbd107(0).CAllow = "C"                                   '
         Kbd107(0).AAllow = "A"                                   '
         Kbd107(0).UpKey = " "                                    '
         Kbd107(0).ForceUp = " "                                  '
         Kbd107(0).Labl = " "                                     '
         Kbd107(0).NData = "(PASSTHRU)"                           '
         Kbd107(0).NUp = " "                                      '
         Kbd107(0).SData = "(PASSTHRU)"                           '
         Kbd107(0).SUp = " "                                      '
         Kbd107(0).CData = "(PASSTHRU)"                           '
         Kbd107(0).CUp = " "                                      '
         Kbd107(0).AData = "(PASSTHRU)"                           '
         Kbd107(0).AUp = " "                                      '
         Kbd107(0).SCData = "(PASSTHRU)"                          '
         Kbd107(0).SCUp = " "                                     '
         Kbd107(0).SAData = "(PASSTHRU)"                          '
         Kbd107(0).SAUp = " "                                     '
         Kbd107(0).SCAData = "(PASSTHRU)"                         '
         Kbd107(0).SCAUp = " "                                    '
         Kbd107(0).CAData = "(PASSTHRU)"                          '
         Kbd107(0).CAUp = " "                                     '

         '-----------------------------------------------------------------------------------------+
         '- Get our Master table from the EXE resource data                                        |
         '-----------------------------------------------------------------------------------------+
         kbData = RESOURCE$(RCDATA, "KBDATA")                     ' Get kbData from our RES file default
         j = PARSECOUNT(kbData, $CRLF)                            ' Loop through the data finally
         FOR kbDix = 1 TO j                                       '
            TX = PARSE$(kbData, $CRLF, kbDix)                     ' Extract a line

            '--------------------------------------------------------------------------------------+
            '- Comment lines                                                                       |
            '--------------------------------------------------------------------------------------+
            IF LEFT$(TX, 1) = ";" OR ISNULL(TRIM$(TX)) THEN       ' Skip comments, reset phase
               phase = 1                                          '
               ITERATE FOR                                        '
            END IF                                                '
            SELECT CASE AS LONG phase                             ' Which line of Set?

               '-----------------------------------------------------------------------------------+
               '- First line of a set                                                              |
               '-----------------------------------------------------------------------------------+
               CASE 1                                             ' First
                  PARSE TX, Args()                                ' Break out the 'words'
                  INCR kIX                                        ' Bump Table index
                  IF ISNOTHING(Kbd107(kIX)) THEN  LET Kbd107(kIX) = CLASS "cKBMastEntry"  ' Assign an object
                  KeyCode(kIX) = CHR$(VAL(Args(0)))               ' Save the parsed data
                  Kbd107(kIX).Extern = LEFT$(Args(1), 1)          '
                  Kbd107(kIX).KIcon = TRIM$(Args(2))              '
                  Kbd107(kIX).W = VAL(Args(3))                    '
                  Kbd107(kIX).H = VAL(Args(4))                    '
                  Kbd107(kIX).X = VAL(Args(5))                    '
                  Kbd107(kIX).Y = VAL(Args(6))                    '
                  Kbd107(kIX).NAllow = MID$(Args(7), 1, 1)        '
                  Kbd107(kIX).SAllow = MID$(Args(7), 2, 1)        '
                  Kbd107(kIX).CAllow = MID$(Args(7), 3, 1)        '
                  Kbd107(kIX).AAllow = MID$(Args(7), 4, 1)        '
                  Kbd107(kIX).UpKey = MID$(Args(7), 5, 1)         '
                  Kbd107(kIX).ForceUp = MID$(Args(7), 6, 1)       '
                  Kbd107(kIX).Labl = Args(8)                      '
                  INCR phase                                      ' On to next phase

               '-----------------------------------------------------------------------------------+
               '- Second line of a set (Normal Key)                                                |
               '-----------------------------------------------------------------------------------+
               CASE 2                                             ' Normal command
                  Kbd107(kIX).NData = MID$(TX, 3)                 ' Save key data
                  Kbd107(kIX).NUp = LEFT$(TX, 1)                  '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Third line of a set (Shift Key)                                                  |
               '-----------------------------------------------------------------------------------+
               CASE 3                                             ' Normal command
                  Kbd107(kIX).SData = MID$(TX, 3)                 ' Save key data
                  Kbd107(kIX).SUp = LEFT$(TX,1)                   '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Fourth line of a set (Control Key)                                               |
               '-----------------------------------------------------------------------------------+
               CASE 4                                             ' Normal command
                  Kbd107(kIX).CData = MID$(TX, 3)                 ' Save key data
                  Kbd107(kIX).CUp = LEFT$(TX, 1)                  '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Fifth line of a set (ALT Key)                                                    |
               '-----------------------------------------------------------------------------------+
               CASE 5                                             ' Normal command
                  Kbd107(kIX).AData = MID$(TX, 3)                 ' Save key data
                  Kbd107(kIX).AUp = LEFT$(TX, 1)                  '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Sixth line of a set (Shift-Control Key)                                          |
               '-----------------------------------------------------------------------------------+
               CASE 6                                             ' Normal command
                  Kbd107(kIX).SCData = MID$(TX, 3)                ' Save key data
                  Kbd107(kIX).SCUp = LEFT$(TX, 1)                 '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Seventh line of a set (Shift-ALT Key)                                            |
               '-----------------------------------------------------------------------------------+
               CASE 7                                             ' Normal command
                  Kbd107(kIX).SAData = MID$(TX, 3)                ' Save key data
                  Kbd107(kIX).SAUp = LEFT$(TX, 1)                 '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Eighth line of a set (Shift-Control-ALT Key)                                     |
               '-----------------------------------------------------------------------------------+
               CASE 8                                             ' Normal command
                  Kbd107(kIX).SCAData = MID$(TX, 3)               ' Save Key Data
                  Kbd107(kIX).SCAUp = LEFT$(TX, 1)                '
                  INCR phase                                      ' Next phase

               '-----------------------------------------------------------------------------------+
               '- Ninth line of a set (Control-ALT Key)                                            |
               '-----------------------------------------------------------------------------------+
               CASE 9                                             ' Normal command
                  Kbd107(kIX).CAData = MID$(TX, 3)                ' Save key data
                  Kbd107(kIX).CAUp = LEFT$(TX, 1)                 '
                  INCR phase                                      ' Next phase
            END SELECT                                            '
         NEXT kbDIX                                               ' Loop through kbData

         '-----------------------------------------------------------------------------------------+
         '- If this is the very initial run, set the user's choice for Enter/NewLine               |
         '-----------------------------------------------------------------------------------------+
         IF gENV.WelcomeOpt > 0 THEN                              ' Is this the initial setup?
            IF gENV.WelcomeOpt = 1 THEN                           ' Enter and Backspace
               Kbd107(44).NData = "(Enter)"                       '
               Kbd107(30).NData = "(NewLine)"                     '
            ELSEIF gENV.WelcomeOpt = 2 THEN                       ' Enter and Ctrl-Enter
               Kbd107(44).NData = "(Enter)"                       '
               Kbd107(44).CData = "(NewLine)"                     '
            ELSEIF gENV.WelcomeOpt = 3 THEN                       ' R-Ctrl and Enter
               Kbd107(77).NData = "(Enter)"                       '
               Kbd107(44).NData = "(NewLine)"                     '
            END IF                                                '
            gKbdT.WriteKBINITUser                                 ' Save things
            gENV.WelcomeOpt = 0                                   ' Clear the WelcomeOpt Flag.
            gENV.SetINITimeStamp                                  '
            EXIT METHOD                                           ' We're done
         END IF                                                   '

         '-----------------------------------------------------------------------------------------+
         '- Get the KBD table                                                                      |
         '-----------------------------------------------------------------------------------------+
         i = 0: j = 0: k = 0                                      ' Reset counters
         gSQL.SelBegin("select * FROM " + $DQ + gSQL.KBDName + $DQ)  ' Ask for everything
         IF gSQL.SelFirst() THEN                                  ' Select the first
            DO                                                    ' Loop through them
               INCR i                                             ' Count the line
               IF i = 1 THEN ITERATE DO                           ' Skip the count line
               IF k = 0 THEN INCR j                               ' Bump KBD table ctr if beginning of a set
               SELECT CASE k                                      ' Now handle the lines of a set
                  CASE 0: INCR k                                  ' Comment line
                  CASE 1: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).UpKey = TX '
                  CASE 2: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).NUp =  LEFT$(tx, 1): Kbd107(j).NData =   MID$(TX, 3) '
                  CASE 3: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).SUp =  LEFT$(tx, 1): Kbd107(j).SData =   MID$(TX, 3) '
                  CASE 4: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).CUp =  LEFT$(tx, 1): Kbd107(j).CData =   MID$(TX, 3) '
                  CASE 5: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).AUp =  LEFT$(tx, 1): Kbd107(j).AData =   MID$(TX, 3) '
                  CASE 6: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).SCUp = LEFT$(tx, 1): Kbd107(j).SCData =  MID$(TX, 3) '
                  CASE 7: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).SAUp = LEFT$(tx, 1): Kbd107(j).SAData =  MID$(TX, 3) '
                  CASE 8: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).SCAUp = LEFT$(tx, 1): Kbd107(j).SCAData = MID$(TX, 3)   '
                  CASE 9: INCR k: TX = TRIM$(gSQL.SelGet("DBData")): Kbd107(j).CAUp = LEFT$(tx, 1): Kbd107(j).CAData =  MID$(TX, 3) '
               END SELECT                                         '
               IF k = 10 THEN k = 0                               ' Reset Set counter
            LOOP WHILE gSQL.SelNext()                             '
         END IF                                                   '
         gSQL.SelEnd                                              ' Close the SQL request
      END METHOD                                                  '

      METHOD LoadKbFinal()                                        '
      LOCAL kIX, aIX, i AS LONG, pfk, pfpfx AS STRING             '
      DIM args(10) AS STRING                                      '


      '--------------------------------------------------------------------------------------------+
      '- Build the final active table                                                              |
      '--------------------------------------------------------------------------------------------+
      KbdPFShowNum = 0                                            '
      KbdPFShowMax = 0                                            '
      aIX = 0                                                     '
      FOR kIX = 0 TO 107                                          ' Loop through the master table
         IF Kbd107(kIX).ForceUp = "D" THEN                        ' Untouchable key?
            ITERATE FOR                                           '
         END IF                                                   '

         IF Kbd107(kIX).NAllow = "N" AND _                        ' Do normal (if not NULL)
            UUCASE(TRIM$(Kbd107(kIX).NData)) <> "(NULL)" THEN     '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "..."   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).NUp) '
            Kbd(aIX).KData = Kbd107(kIX).NData                    '       "   "   Data
            pfpfx = "1": GOSUB AddPFShow                          ' Set prefix and go add PFShow data
            IF KeyCode(kIX) = CHR$(%VK_RETURN) THEN KbdKPEnter = aIX ' Save index of Enter key
         END IF                                                   '

         IF Kbd107(kIX).SAllow = "S" AND _                        ' Do Shifted (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).SData)) <> "(NULL)" THEN    '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "S.."   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).SUp) '
            Kbd(aIX).KData = Kbd107(kIX).SData                    '   "   Data
            pfpfx = "2S-": GOSUB AddPFShow                        ' Set prefix and go add PFShow data
         END IF                                                   '
                                                                  '
         IF Kbd107(kIX).CAllow = "C" AND _                        ' Do Control (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).CData)) <> "(NULL)" THEN    '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + ".C."   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).CUp) '
            Kbd(aIX).KData = Kbd107(kIX).CData                    '   "   Data
            pfpfx = "3C-": GOSUB AddPFShow                        ' Set prefix and go add PFShow data
         END IF                                                   '

         IF Kbd107(kIX).AAllow = "A" AND _                        ' Do Alt (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).AData)) <> "(NULL)" THEN    '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "..A"   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).AUp) '
            Kbd(aIX).KData = Kbd107(kIX).AData                    '   "   Data
            pfpfx = "4A-": GOSUB AddPFShow                        ' Set prefix and go add PFShow data
         END IF                                                   '

         IF Kbd107(kIX).SAllow = "S" AND Kbd107(kIX).CAllow = "C" AND _ ' Do Shift-Control (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).SCData)) <> "(NULL)" THEN   '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "SC."   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).SCUp)   '
            Kbd(aIX).KData = Kbd107(kIX).SCData                   '   "   Data
            pfpfx = "5SC-": GOSUB AddPFShow                       ' Set prefix and go add PFShow data

         END IF                                                   '

         IF Kbd107(kIX).SAllow = "S" AND Kbd107(kIX).AAllow = "A" AND _ ' Do Shift-Alt (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).SAData)) <> "(NULL)" THEN   '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "S.A"   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).SAUp)   '
            Kbd(aIX).KData = Kbd107(kIX).SAData                   '   "   Data
            pfpfx = "7SA-": GOSUB AddPFShow                       ' Set prefix and go add PFShow data
         END IF                                                   '

         IF Kbd107(kIX).SAllow = "S" AND Kbd107(kIX).CAllow = "C" AND Kbd107(kIX).AAllow = "A" AND _  ' Do Shift-Control-Alt (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).SCAData)) <> "(NULL)" THEN  '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + "SCA"   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).SCAUp)  '
            Kbd(aIX).KData = Kbd107(kIX).SCAData                  '   "   Data
            pfpfx = "8SCA-": GOSUB AddPFShow                      ' Set prefix and go add PFShow data
         END IF                                                   '

         IF Kbd107(kIX).CAllow = "C" AND Kbd107(kIX).AAllow = "A" AND _ ' Do Control-Alt (if not NULL)
             UUCASE(TRIM$(Kbd107(kIX).CAData)) <> "(NULL)" THEN   '
            INCR aIX                                              ' Incr active table index
            IF ISNOTHING(Kbd(aIX)) THEN LET Kbd(aIX) = CLASS "cKBActive"   ' Assign an object
            KMode(aIX) = KeyCode(kIX) + Kbd107(kIX).Extern + ".CA"   '
            Kbd(aIX).SupSent = "N"                                ' 1st Sent
            Kbd(aIX).SupRepeat = IIF$(Kbd107(kIX).ForceUp = "U", "U", Kbd107(kIX).CAUp)   '
            Kbd(aIX).KData = Kbd107(kIX).CAData                   '   "   Data
            pfpfx = "6CA-": GOSUB AddPFShow                       ' Set prefix and go add PFShow data
         END IF                                                   '

      NEXT kIX                                                    '
      KbdCount = aIX + 1                                          ' Save size of table
      ARRAY SORT KbdPFShow() FOR (KbdPFShowNum + 1), CALL KbdSortExit() ' Sort PFShow able
      KbdPFShowMax = MAX(8, KbdPFShowMax)                         ' Just to avoid div by zero
      KbdPFShowMax = MIN(20, KbdPFShowMax)                        ' To avoid being too big

      EXIT METHOD                                                 '

      AddPFShow:                                                  '
         IF LEFT$(Kbd(aIX).KData, 1) = "<" THEN                   ' Possibly labelled?
            IF UCASE$(LEFT$(Kbd(aIX).KData, 4)) = "<IF." THEN RETURN ' If a Conditional test, ignore it
            i = INSTR(Kbd(aIX).KData, ">")                        ' Find the end of the comment
            IF i THEN                                             ' Got it
               INCR KbdPFShowNum                                  ' Incr count
               pfk = pfpfx + Kbd107(kIX).Labl                     ' Get an upper128 version of the text
               StrAdd128(pfk)                                     '
               KbdPFShow(KbdPFShowNum) = pfk + "=" + MID$(Kbd(aIX).KData, 2, i - 2) '
               KbdPFShowMax = MAX(KbdPFShowMax, LEN(KbdPFShow(KbdPFShowNum))) 'Save MAX length
            END IF                                                '
         END IF                                                   '
         RETURN                                                   '

      END METHOD                                                  '

      METHOD WriteKBUser()                                        '
      '--------------------------------------------------------------------------------------------+
      '- Save the gKbd107 table                                                                    |
      '--------------------------------------------------------------------------------------------+
      LOCAL i, j AS LONG                                          '
         MEntry                                                   '
         '-----------------------------------------------------------------------------------------+
         '- Write the users custom KB settings out                                                 |
         '-----------------------------------------------------------------------------------------+
         gSQL.Execute("Start transaction")                        ' To optimize things
         gSQL.UpdateString("K", "R00000", "1070")                 ' Write the Total # entries
         FOR i = 1 TO 107                                         ' Dump the Table
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), "; ------------- " + Kbd107(i).Labl)  ' Delimiter
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).UpKey))   ' UpKey
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).NUp   + "/" + Kbd107(i).NData))   ' Normal
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SUp   + "/" + Kbd107(i).SData))   ' Shift
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).CUp   + "/" + Kbd107(i).CData))   ' Control
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).AUp   + "/" + Kbd107(i).AData))   ' Alt
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SCUp  + "/" + Kbd107(i).SCData))  ' Shft-Ctrl
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SAUp  + "/" + Kbd107(i).SAData))  ' Shft-Alt
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SCAUp + "/" + Kbd107(i).SCAData)) ' Shft-Ctrl-Alt
            INCR j: gSQL.UpdateString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).CAUp  + "/" + Kbd107(i).CAData))  ' Ctrl-Alt
         NEXT i                                                   '
         gSQL.Execute("End transaction")                          ' To optimize things
         MExitMeth                                                '
      END METHOD                                                  '

      METHOD WriteKBInitUser()                                    '
      '--------------------------------------------------------------------------------------------+
      '- Save the gKbd107 table for the first time                                                 |
      '--------------------------------------------------------------------------------------------+
      LOCAL i, j AS LONG                                          '
         MEntry                                                   '
         '-----------------------------------------------------------------------------------------+
         '- Write the users custom KB settings out                                                 |
         '-----------------------------------------------------------------------------------------+
         gSQL.Execute("PRAGMA synchronous = OFF")                 ' To optimize things
         gSQL.AddString("K", "R00000", "1070")                    ' Write the Total # entries
         FOR i = 1 TO 107                                         ' Dump the Table
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), "; ------------- " + Kbd107(i).Labl)  ' Delimiter
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).UpKey))   ' UpKey
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).NUp  + "/" + Kbd107(i).NData)) ' Normal
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SUp  + "/" + Kbd107(i).SData)) ' Shift
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).CUp  + "/" + Kbd107(i).CData)) ' Control
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).AUp  + "/" + Kbd107(i).AData)) ' Alt
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SCUp + "/" + Kbd107(i).SCData))   ' Shft-Ctrl
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SAUp + "/" + Kbd107(i).SAData))   ' Shft-Alt
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).SCAUp + "/" + Kbd107(i).SCAData)) ' Shft-Ctrl-Alt
            INCR j: gSQL.AddString("K", "R" + FORMAT$(j, "00000"), TRIM$(Kbd107(i).CAUp + "/" + Kbd107(i).CAData))   ' Ctrl-Alt
         NEXT i                                                   '
         gSQL.Execute("PRAGMA synchronous = ON")                  '
         MExitMeth                                                '
      END METHOD                                                  '

   END INTERFACE                                                  '

END CLASS                                                         '

'--------------------------------------------------------------------------------------------------+
'- Keyboard Table Class definitions                                                                |
'--------------------------------------------------------------------------------------------------+

CLASS cKBMastEntry                                                '
   INSTANCE KIcon           AS STRING                             ' Name of the Icon in the resource file
   INSTANCE W               AS LONG                               ' Icon width
   INSTANCE H               AS LONG                               ' Icon height
   INSTANCE X               AS LONG                               ' Icon X location
   INSTANCE Y               AS LONG                               ' Icon Y location
   INSTANCE Extern          AS STRING                             ' Extended status
   INSTANCE NAllow          AS STRING                             ' Allowable Normal key
   INSTANCE SAllow          AS STRING                             ' Allowable Shift key
   INSTANCE CAllow          AS STRING                             ' Allowable Control key
   INSTANCE AAllow          AS STRING                             ' Allowable Alt key
   INSTANCE UpKey           AS STRING                             ' Allowable UpKey detect only
   INSTANCE ForceUp         AS STRING                             ' Forced UpKey / Disable All
   INSTANCE NUp             AS STRING                             ' Normal  Up Only
   INSTANCE SUp             AS STRING                             ' Shift   Up Only
   INSTANCE CUp             AS STRING                             ' Control Up Only
   INSTANCE AUp             AS STRING                             ' Alt     Up only
   INSTANCE SCUp            AS STRING                             ' Shift Control Up Only
   INSTANCE SAUp            AS STRING                             ' Shift Alt Up Only
   INSTANCE SCAUp           AS STRING                             ' Shift Control Alt Up Only
   INSTANCE CAUp            AS STRING                             ' Control Alt Up Only
   INSTANCE Labl            AS STRING                             ' External name
   INSTANCE NData           AS STRING                             ' Normal  Data
   INSTANCE SData           AS STRING                             ' Shift   Data
   INSTANCE CData           AS STRING                             ' Control Data
   INSTANCE AData           AS STRING                             ' Alt     Data
   INSTANCE SCData          AS STRING                             ' Shift Control     Data
   INSTANCE SAData          AS STRING                             ' Shift Alt         Data
   INSTANCE SCAData         AS STRING                             ' Shift Control Alt Data
   INSTANCE CAData          AS STRING                             ' Control Alt       Data

   INTERFACE iKBMastEntry: INHERIT IUNKNOWN                       '
      gsProp(KIcon, STRING)                                       ' Icon Name
      gsProp(W,LONG)                                              ' Icon width
      gsProp(H,LONG)                                              ' Icon height
      gsProp(X,LONG)                                              ' Icon X location
      gsProp(Y,LONG)                                              ' Icon Y location
      gsProp(Extern,STRING)                                       ' Extended status
      gsProp(NAllow,STRING)                                       ' Allowable Normal key
      gsProp(SAllow,STRING)                                       ' Allowable Shift key
      gsProp(CAllow,STRING)                                       ' Allowable Control key
      gsProp(AAllow,STRING)                                       ' Allowable Alt key
      gsProp(UpKey,STRING)                                        ' Allowable UpKey detect only
      gsProp(ForceUp,STRING)                                      ' Forced UpKey / Disable All
      gsProp(NUp,STRING)                                          ' Normal  Up Only
      gsProp(SUp,STRING)                                          ' Shift   Up Only
      gsProp(CUp,STRING)                                          ' Control Up Only
      gsProp(AUp,STRING)                                          ' Alt     Up only
      gsProp(SCUp,STRING)                                         ' Shift Control Up Only
      gsProp(SAUp,STRING)                                         ' Shift Alt Up Only
      gsProp(SCAUp,STRING)                                        ' Shift Control Alt Up Only
      gsProp(CAUp,STRING)                                         ' Control Alt Up Only
      gsProp(Labl,STRING)                                         ' External name
      gsProp(NData,STRING)                                        ' Normal  Data
      gsProp(SData,STRING)                                        ' Shift   Data
      gsProp(CData,STRING)                                        ' Control Data
      gsProp(AData,STRING)                                        ' Alt     Data
      gsProp(SCData,STRING)                                       ' Shift Control     Data
      gsProp(SAData,STRING)                                       ' Shift Alt         Data
      gsProp(SCAData,STRING)                                      ' Shift Control Alt Data
      gsProp(CAData,STRING)                                       ' Control Alt       Data
   END INTERFACE                                                  '
END CLASS                                                         '

CLASS cKBactive                                                   '
   INSTANCE SupRepeat       AS STRING                             ' Suppress repeat (Up signal only)
   INSTANCE SupSent         AS STRING                             ' Suppressed, 1st has been sent
   INSTANCE KData           AS STRING                             ' Keyboard data string
   INTERFACE iKBActive: INHERIT IUNKNOWN                          '
      gsProp(SupRepeat,STRING)                                    ' Suppress repeat (Up signal only)
      gsProp(SupSent,STRING)                                      ' Suppressed, 1st has been sent
      gsProp(KData,STRING)                                        ' Keyboard data string
   END INTERFACE                                                  '
END CLASS                                                         '
