'--------------------------------------------------------------------------------------------------+
'- License Stuff                                                                                   |
'-                                                                                                 |
'-                                                                                                 |
'-   SPFLite is free software: you can redistribute it and/or modify                               |
'-   it under the terms of the GNU General Public License as published by                          |
'-   the Free Software Foundation, either version 3 of the License, or                             |
'-   (at your option) any later version.                                                           |
'-                                                                                                 |
'-   SPFLite is distributed in the hope that it will be useful,                                    |
'-   but WITHOUT ANY WARRANTY; without even the implied warranty of                                |
'-   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                 |
'-   GNU General Public License for more details.                                                  |
'-                                                                                                 |
'-   You should have received a copy of the GNU General Public License                             |
'-   along with SPFLite.  If not, see <https://www.gnu.org/licenses/>.                             |
'-                                                                                                 |
'--------------------------------------------------------------------------------------------------+
'- ObjSQL.inc                                                                                      |
'--------------------------------------------------------------------------------------------------+
                                                                  '
                                                                  '
CLASS SQLC                                                        '
'--------------------------------------------------------------------------------------------------+
'- Data                                                                                            |
'--------------------------------------------------------------------------------------------------+
INSTANCE hDB            AS LONG                                   '
INSTANCE TablePtr       AS LONG PTR                               '
INSTANCE tColumns()     AS STRING                                 '
INSTANCE tRowCount      AS LONG                                   '
INSTANCE tColCount      AS LONG                                   '
INSTANCE tRowNum        AS LONG                                   '
INSTANCE tFirstCol      AS LONG                                   '

INSTANCE OptName        AS STRING                                 ' Active Options  Table
INSTANCE KbdName        AS STRING                                 ' Active Keyboard Table
INSTANCE BkpName        AS STRING                                 ' Active Backup   Table
INSTANCE RtrName        AS STRING                                 ' Active Retrieve Table
INSTANCE SetName        AS STRING                                 ' Active Set      Table
INSTANCE EFTName        AS STRING                                 ' Active EFT      Table

CLASS METHOD CREATE()                                             '
'--------------------------------------------------------------------------------------------------+
'- CFG initialization                                                                              |
'--------------------------------------------------------------------------------------------------+
   OptName = "ODEFAULT"                                           ' Set default Options  Table
   KbdName = "KDEFAULT"                                           ' Set default Keyboard Table
   BkpName = "BDEFAULT"                                           ' Set default Backup   Table
   RtrName = "RDEFAULT"                                           ' Set default Retrieve Table
   SetName = "SDEFAULT"                                           ' Set default SET      Table
   EFTName = "EDEFAULT"                                           ' Set default EFT      Table
END METHOD                                                        '

INTERFACE SQLI                                                    '

   INHERIT IUNKNOWN                                               '

      GProp(OptName, STRING)                                      ' Return active Options  table name
      PROPERTY SET OptName(v AS STRING): OptName = "O" + UCASE$(v): END PROPERTY '
      GProp(KbdName, STRING)                                      ' Return active Keyboard table name
      PROPERTY SET KbdName(v AS STRING): KbdName = "K" + UCASE$(v): END PROPERTY '
      GProp(BkpName, STRING)                                      ' Return active Backup   table name
      PROPERTY SET BKPName(v AS STRING): BkpName = "B" + UCASE$(v): END PROPERTY '
      GProp(RtrName, STRING)                                      ' Return active Retrieve table name
      PROPERTY SET RtrName(v AS STRING): RtrName = "R" + UCASE$(v): END PROPERTY '
      GProp(SetName, STRING)                                      ' Return active SET      table name
      PROPERTY SET SetName(v AS STRING): SetName = "S" + UCASE$(v): END PROPERTY '
      GProp(EFTName, STRING)                                      ' Return active EFT      table name
      PROPERTY SET EFTName(v AS STRING): EFTName = "E" + UCASE$(v): END PROPERTY '

   METHOD DBClose()                                               '
   '-----------------------------------------------------------------------------------------------+
   '- Close database file                                                                          |
   '-----------------------------------------------------------------------------------------------+
      sqlite3_close(hDB)                                          ' Close it
      hDB = 0                                                     ' Clear Handle
   END METHOD                                                     '

   METHOD DBOpen(file AS STRING) AS LONG                          '
   '-----------------------------------------------------------------------------------------------+
   '- Open the DB                                                                                  |
   '-----------------------------------------------------------------------------------------------+
   LOCAL UTFFile AS WSTRING, i AS LONG                            '
      METHOD = %False:                                            ' Default return = failure
      UTFFile = file                                              ' Get filename into UTF-16
      i = sqlite3_open16(BYVAL STRPTR(UtFFile), hDB)              ' Try the Open
      IF i = 0 THEN                                               ' Well?
         METHOD = %True                                           ' Open OK
         me.Execute("Pragma journal_mode = memory")               ' Set journal to truncate mode
         me.Execute("Pragma page_size = 1024")                    ' Set page size
      END IF                                                      '
   END METHOD                                                     '

   METHOD Execute(sql AS STRING) AS LONG                          '
   '-----------------------------------------------------------------------------------------------+
   '- Execute an SQL statement with NO return                                                      |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lclTable    AS LONG PTR                                  ' Local data
   LOCAL lclErrorSz, lclRowCount, lclColCount, RC AS LONG         '
      RC = sqlite_get_table(hDB, BYVAL STRPTR(sql), lclTable, lclRowCount, lclColCount, lclErrorSz)   '
      METHOD = IIF(RC = 0, %True, %False )                        '
      sqlite_free_table lclTable                                  '
   END METHOD                                                     '

   METHOD FIX(BYVAL s AS STRING) AS STRING                        '
   '-----------------------------------------------------------------------------------------------+
   '- Escape single quotes in string                                                               |
   '-----------------------------------------------------------------------------------------------+
      REPLACE "'" WITH "''" IN s                                  ' DOUBLE up SINGLE quotes
      s = StrAnsi2Utf8(s)                                         ' UTF8 Encode it
      METHOD = s                                                  ' Return it
   END METHOD                                                     '

   METHOD AddString(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Add a Key/Data pair to the Config DB                                                         |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert into " + me.TableName(sTable) + " values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')") '
   END METHOD                                                     '

   METHOD AddStringDirect(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Add a Key/Data pair to the Config DB                                                         |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert into " + $DQ + sTable + $DQ + " values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')")   '
   END METHOD                                                     '

   METHOD GetString(sTable AS STRING, sKey AS STRING, sDefault AS STRING) AS STRING '
   '-----------------------------------------------------------------------------------------------+
   '- Get string from SQL file or return default (and save default as new entry)                   |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lStr AS STRING                                           '
      IF me.Quickie("select DBData from " + me.TableName(sTable) + " where DBKey = '" + UCASE$(sKey) + "'", lStr) THEN  '
         METHOD = lStr                                            ' Return the answer
      ELSE                                                        ' Otherwise
         METHOD = sDefault                                        ' Return the default
         IF sDefault <> "*MISSING*" THEN me.SetString(sTable, UCASE$(sKey), sDefault)  ' And save the default
      END IF                                                      '
   END METHOD                                                     '

   METHOD GetStringDirect(sTable AS STRING, sKey AS STRING, sDefault AS STRING) AS STRING '
   '-----------------------------------------------------------------------------------------------+
   '- Get string from SQL file                                                                     |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lStr AS STRING                                           '
      IF me.Quickie("select DBData from " + $DQ + sTable + $DQ + " where DBKey = '" + UCASE$(sKey) + "'", lStr) THEN '
         METHOD = lStr                                            ' Return the answer
      ELSE                                                        ' Otherwise
         METHOD = sDefault                                        ' Return the default
         IF sDefault <> "*MISSING*" THEN me.SetStringDirect(sTable, UCASE$(sKey), sDefault)  ' And save the default
      END IF                                                      '
   END METHOD                                                     '

   METHOD LastError() AS STRING                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Return last error message                                                                    |
   '-----------------------------------------------------------------------------------------------+
   LOCAL pzErr AS ASCIIZ PTR                                      ' String pointer
      pzErr = sqlite3_errmsg(hDB)                                 ' Ask SQLite for it
      IF pzErr THEN METHOD = @pzErr                               ' If returned, pass back the string
   END METHOD                                                     '

   METHOD Quickie(sql AS STRING, Ans AS STRING) AS LONG           '
   '-----------------------------------------------------------------------------------------------+
   '- Do a quick SQLquery                                                                          |
   '-----------------------------------------------------------------------------------------------+
      IF me.SelBegin(sql) AND me.SelFirst() THEN                  ' Set start and get first
         Ans = me.SelGetAt(1)                                     ' OK, return values
         METHOD = %True                                           ' And True
      ELSE                                                        ' Else
         Ans = "": METHOD = %False                                ' Return Null and fail
      END IF                                                      '
      me.SelEnd()                                                 ' Always end the select
   END METHOD                                                     '

   METHOD RowDelete(sTable AS STRING, sKey AS STRING) AS LONG     '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a single row                                                                          |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("delete from " + me.TableName(sTable) + " where DBKey = '" + UCASE$(sKey) + "'")  '
   END METHOD                                                     '

   METHOD RowDeleteDirect(sTable AS STRING, sKey AS STRING) AS LONG  '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a single row                                                                          |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("delete from " + $DQ + sTable + $DQ + " where DBKey = '" + UCASE$(sKey) + "'") '
   END METHOD                                                     '

   METHOD SelBegin(sql AS STRING) AS LONG                         '
   '-----------------------------------------------------------------------------------------------+
   '- Execute a SQLquery and return a record set                                                   |
   '-----------------------------------------------------------------------------------------------+
   LOCAL i, RC AS LONG                                            '
   LOCAL lclErrorSz  AS LONG                                      '
   LOCAL lclCol AS ASCIIZ PTR                                     '
      RESET TablePtr, tRowCount, tColCount, tRowNum, tFirstCol    '
      ERASE tColumns()                                            '
      IF LEN(sql) THEN                                            '
         IF sqlite_get_table(hDB, BYVAL STRPTR(sql), TablePtr, tRowCount, tColCount, lclErrorSz) = 0 THEN   '
            IF tRowCount = 0 THEN                                 ' Anything come back?
               METHOD = %True                                     ' No, just return True
               sqlite_free_table TablePtr                         ' Free the table
            ELSE                                                  ' Something came back
               REDIM tColumns(1 TO tColCount)                     ' Redim tableto fit
               FOR i = 1 TO tColCount                             ' Populate it
                  lclCol = @TablePtr[i - 1]                       '
                  tColumns(i) = @lclCol                           '
               NEXT i                                             '
               METHOD = %True                                     ' Return True
            END IF                                                '
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelColumn(columnNo AS LONG) AS STRING                   '
   '-----------------------------------------------------------------------------------------------+
   '- Get column name for columnNo in SQLquery results, ONE based index                            |
   '-----------------------------------------------------------------------------------------------+
      IF columnNo > 0 AND columnNo <= UBOUND(tColumns) THEN METHOD = tColumns(columnNo)   '
   END METHOD                                                     '

   METHOD SelColumnNo(sColumn AS STRING) AS LONG                  '
   '-----------------------------------------------------------------------------------------------+
   '- Return position of column in SQLquery results, zero if not found                             |
   '-----------------------------------------------------------------------------------------------+
   LOCAL x AS LONG                                                ' Local Data
      ARRAY SCAN tColumns(), COLLATE UCASE, = sColumn, TO x       ' Search for it
      METHOD = x                                                  ' Return answer
   END METHOD                                                     '

   METHOD SelEnd()                                                '
   '-----------------------------------------------------------------------------------------------+
   '- Close SQLquery                                                                               |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN sqlite_free_table TablePtr                ' If we got something, Free it
      RESET TablePtr, tRowCount, tColCount, tRowNum, tFirstCol    ' Clear the data fields
      ERASE tColumns()                                            '
   END METHOD                                                     '

   METHOD SelFirst() AS LONG                                      '
   '-----------------------------------------------------------------------------------------------+
   '- Move to first row in SQLquery results                                                        |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         tRowNum = 1                                              ' Set to Row 1
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGet(sColumn AS STRING) AS STRING                     '
   '-----------------------------------------------------------------------------------------------+
   '- Get value for column on current row of SQLquery results                                      |
   '-----------------------------------------------------------------------------------------------+
   LOCAL x, ndx, UTFError AS LONG, ANSIAnswer AS STRING           ' Local data
   LOCAL lclCol AS ASCIIZ PTR                                     '
      IF tRowNum > 0 AND tRowNum <= tRowCount THEN                ' If on a valid Row number
         x = me.SelColumnNo(sColumn)                              ' Get position of column in results
         ndx = tFirstCol + x - 1                                  ' Calc Index
         lclCol = @TablePtr[ndx]                                  ' Pick up column
         ANSIAnswer = @lclCol                                     ' ASCIIZ to STRING
         METHOD = StrUtf82Ansi(ANSIAnswer, UTFError)              ' Return the string
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGetAt(columnNo AS LONG) AS STRING                    '
   '-----------------------------------------------------------------------------------------------+
   '- Get value for columnNo on current row of SQLquery results                                    |
   '-----------------------------------------------------------------------------------------------+
    LOCAL ndx, UTFError AS LONG, ANSIAnswer AS STRING             ' Local Data
    LOCAL lclCol AS ASCIIZ PTR                                    '
      IF tRowNum > 0 AND tRowNum <= tRowCount THEN                ' If valid Row number
         IF columnNo > 0 AND columnNo <= UBOUND(tColumns) THEN    ' and valid Column number
            ndx = tFirstCol + columnNo - 1                        ' Calc index
            lclCol = @TablePtr[ndx]                               ' Pick up the column
            ANSIAnswer = @lclCol                                  ' ASCIIZ to STRING
         METHOD = StrUtf82Ansi(ANSIAnswer, UTFError)              ' Return the string
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGoTo(srow AS LONG) AS LONG                           '
   '-----------------------------------------------------------------------------------------------+
   '- Move to row in SQLquery results                                                              |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount AND srow > 0 AND srow <= tRowCount THEN        ' If requested row is valid
         tRowNum = srow                                           ' Go to it
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelLast() AS LONG                                       '
   '-----------------------------------------------------------------------------------------------+
   '- Move to last row in SQLquery results                                                         |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         tRowNum = tRowCount                                      ' Set to last row
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelNext() AS LONG                                       '
   '-----------------------------------------------------------------------------------------------+
   '- Move to next row in SQLquery results                                                         |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         INCR tRowNum                                             ' Bump Row number
         IF tRowNum > 0 AND tRowNum <= tRowCount THEN             ' If not past the end
            tFirstCol = tRowNum * tColCount                       ' Set FirstCol
            METHOD = %True                                        ' Return True
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelPrevious() AS LONG                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Move to previous row in SQLquery results                                                     |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         DECR tRowNum                                             ' Drop by 1
         IF tRowNum > 0 AND tRowNum <= tRowCount THEN             ' If not too low
            tFirstCol = tRowNum * tColCount                       ' Set FirstCol
            METHOD = %True                                        ' Return True
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SetString(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Add/Replace a Key/Data pair to the Config DB                                                 |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert or replace into " + me.TableName(sTable) + " values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')")  '
   END METHOD                                                     '

   METHOD SetStringDirect(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Get string from SQL file                                                                     |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert or replace into " + $DQ + sTable + $DQ + " values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')") '
   END METHOD                                                     '

   METHOD TableCopy(sSource AS STRING, sDest AS STRING) AS LONG   '
   '-----------------------------------------------------------------------------------------------+
   '- Copy one table to a new one                                                                  |
   '-----------------------------------------------------------------------------------------------+
   LOCAL i AS LONG                                                '
      i = me.Execute("create table if not exists " + $DQ + UCASE$(sDest) + $DQ + _  '
                    " (DBKey text primary key unique collate nocase, DBData text)") '
      IF i THEN _                                                 '
         i = me.Execute("Insert into " + $DQ + UCASE$(sDest) + $DQ + " select * from " + $DQ + UCASE$(sSource) + $DQ)   '
      METHOD = i                                                  '
   END METHOD                                                     '

   METHOD TableClear(sTable AS STRING) AS LONG                    '
   '-----------------------------------------------------------------------------------------------+
   '- Clear a Table                                                                                |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("delete from " + $DQ + UCASE$(sTable) + $DQ)  ' Clear all rows
   END METHOD                                                     '

   METHOD TableCreate(TableName AS STRING) AS LONG                '
   '-----------------------------------------------------------------------------------------------+
   '- Create a Db Table                                                                            |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("create table if not exists " + $DQ + TableName + $DQ  + _   '
                    " (DBKey text primary key unique collate nocase, DBData text)") '
   END METHOD                                                     '

   METHOD TableDelete(sTable AS STRING) AS LONG                   '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a Table                                                                               |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("drop table if exists " + $DQ + UCASE$(sTable) + $DQ)  ' Kill the table
   END METHOD                                                     '

   METHOD TableExist(sTable AS STRING) AS LONG                    '
   '-----------------------------------------------------------------------------------------------+
   '- See if a Table exists                                                                        |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lclName AS STRING                                        '
      IF me.Quickie("select name FROM sqlite_master WHERE type ='table' AND name = '" + UCASE$(sTable) + "'", lclName) THEN   '
         METHOD = %True                                           ' Say True if OK
      END IF                                                      '
   END METHOD                                                     '

   METHOD TableName(tname AS STRING) AS STRING                    '
   '-----------------------------------------------------------------------------------------------+
   '- Return proper table name                                                                     |
   '-----------------------------------------------------------------------------------------------+
   LOCAL uName AS STRING                                          '
      uname = UCASE$(tname)                                       ' Uppercase just in case
      METHOD = SWITCH$(uname = "O", $DQ + OptName + $DQ, _        ' Return Active table name
                       uname = "B", $DQ + BkpName + $DQ, _        '
                       uname = "R", $DQ + RtrName + $DQ, _        '
                       uname = "S", $DQ + SetName + $DQ, _        '
                       uname = "K", $DQ + KbdName + $DQ, _        '
                       uname = "E", $DQ + EFTName + $DQ)          '
   END METHOD                                                     '

   METHOD TableRename(sTable AS STRING, nTable AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a Table                                                                               |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("alter table " + $DQ + UCASE$(sTable) + $DQ + " rename to " + $DQ + UCASE$(nTable) + $DQ)  ' Rename  the table
   END METHOD                                                     '

   METHOD UpdateString(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Update a table value                                                                         |
   '-----------------------------------------------------------------------------------------------+
       METHOD = me.execute("update " + me.TableName(sTable) + " SET DBData = '" + me.Fix(sStr) + "' where DBKey = '" + UCASE$(sKey) + "'")   '
   END METHOD                                                     '

   METHOD UpdateStringDirect(sTable AS STRING, sKey AS STRING, sStr AS STRING) AS LONG '
   '-----------------------------------------------------------------------------------------------+
   '- Update a table value                                                                         |
   '-----------------------------------------------------------------------------------------------+
       METHOD = me.execute("update " + $DQ + sTable + $DQ + " SET DBData = '" + me.Fix(sStr) + "' where DBKey = '" + UCASE$(sKey) + "'")  '
   END METHOD                                                     '

   METHOD VacuumDB()                                              '
   '-----------------------------------------------------------------------------------------------+
   '- Vacuum the database file                                                                     |
   '-----------------------------------------------------------------------------------------------+
      me.Execute("vacuum")                                        ' Vacuum it
   END METHOD                                                     '

   END INTERFACE                                                  '
END CLASS                                                         '
