'--------------------------------------------------------------------------------------------------+
'- License Stuff                                                                                   |
'-                                                                                                 |
'-                                                                                                 |
'-   SPFLite is free software: you can redistribute it and/or modify                               |
'-   it under the terms of the GNU General Public License as published by                          |
'-   the Free Software Foundation, either version 3 of the License, or                             |
'-   (at your option) any later version.                                                           |
'-                                                                                                 |
'-   SPFLite is distributed in the hope that it will be useful,                                    |
'-   but WITHOUT ANY WARRANTY; without even the implied warranty of                                |
'-   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                                 |
'-   GNU General Public License for more details.                                                  |
'-                                                                                                 |
'-   You should have received a copy of the GNU General Public License                             |
'-   along with SPFLite.  If not, see <https://www.gnu.org/licenses/>.                             |
'-                                                                                                 |
'--------------------------------------------------------------------------------------------------+
'- ObjSQLM.inc                                                                                     |
'--------------------------------------------------------------------------------------------------+
                                                                  '
                                                                  '
CLASS SQLMC                                                       '
'--------------------------------------------------------------------------------------------------+
'- Data                                                                                            |
'--------------------------------------------------------------------------------------------------+
INSTANCE hDB            AS LONG                                   '
INSTANCE TablePtr       AS LONG PTR                               '
INSTANCE tColumns()     AS STRING                                 '
INSTANCE tRowCount      AS LONG                                   '
INSTANCE tColCount      AS LONG                                   '
INSTANCE tRowNum        AS LONG                                   '
INSTANCE tFirstCol      AS LONG                                   '

INTERFACE SQLMI                                                   '

   INHERIT IUNKNOWN                                               '

   METHOD DBClose()                                               '
   '-----------------------------------------------------------------------------------------------+
   '- Close database file                                                                          |
   '-----------------------------------------------------------------------------------------------+
      sqlite3_close(hDB)                                          ' Close it
      hDB = 0                                                     ' Clear Handle
   END METHOD                                                     '

   METHOD DBOpen() AS LONG                                        '
   '-----------------------------------------------------------------------------------------------+
   '- Open the Memory DB                                                                           |
   '-----------------------------------------------------------------------------------------------+
   LOCAL UTFFile AS WSTRING, i AS LONG                            '
      METHOD = %False:                                            ' Default return = failure
      UTFFile = ":memory:"                                        ' Magic word for a Memory DB
      i = sqlite3_open16(BYVAL STRPTR(UtFFile), hDB)              ' Try the Open
      IF i = 0 THEN                                               ' Well?
         METHOD = %True                                           ' Open OK
      END IF                                                      '
   END METHOD                                                     '

   METHOD Execute(sql AS STRING) AS LONG                          '
   '-----------------------------------------------------------------------------------------------+
   '- Execute an SQL statement with NO return                                                      |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lclTable    AS LONG PTR                                  ' Local data
   LOCAL lclErrorSz, lclRowCount, lclColCount, RC AS LONG         '
      RC = sqlite_get_table(hDB, BYVAL STRPTR(sql), lclTable, lclRowCount, lclColCount, lclErrorSz)   '
      METHOD = IIF(RC = 0, %True, %False )                        '
      sqlite_free_table lclTable                                  '
   END METHOD                                                     '

   METHOD FIX(BYVAL s AS STRING) AS STRING                        '
   '-----------------------------------------------------------------------------------------------+
   '- Escape single quotes in string                                                               |
   '-----------------------------------------------------------------------------------------------+
      REPLACE "'" WITH "''" IN s                                  ' DOUBLE up SINGLE quotes
      s = StrAnsi2Utf8(s)                                         ' UTF8 Encode it
      METHOD = s                                                  ' Return it
   END METHOD                                                     '

   METHOD AddString(sKey AS STRING, sStr AS STRING) AS LONG       '
   '-----------------------------------------------------------------------------------------------+
   '- Add a Key/Data pair                                                                          |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert into DataTable values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')")  '
   END METHOD                                                     '

   METHOD GetString(sKey AS STRING) AS STRING                     '
   '-----------------------------------------------------------------------------------------------+
   '- Get string from SQL table                                                                    |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lStr AS STRING                                           '
      IF me.Quickie("select DBData from DataTable where DBKey = '" + UCASE$(sKey) + "'", lStr) THEN   '
         METHOD = lStr                                            ' Return the answer
      ELSE                                                        ' Otherwise
         METHOD = ""                                              ' Return null
      END IF                                                      '
   END METHOD                                                     '

   METHOD LastError() AS STRING                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Return last error message                                                                    |
   '-----------------------------------------------------------------------------------------------+
   LOCAL pzErr AS ASCIIZ PTR                                      ' String pointer
      pzErr = sqlite3_errmsg(hDB)                                 ' Ask SQLite for it
      IF pzErr THEN METHOD = @pzErr                               ' If returned, pass back the string
   END METHOD                                                     '

   METHOD Quickie(sql AS STRING, Ans AS STRING) AS LONG           '
   '-----------------------------------------------------------------------------------------------+
   '- Do a quick SQLquery                                                                          |
   '-----------------------------------------------------------------------------------------------+
      IF me.SelBegin(sql) AND me.SelFirst() THEN                  ' Set start and get first
         Ans = me.SelGetAt(1)                                     ' OK, return values
         METHOD = %True                                           ' And True
      ELSE                                                        ' Else
         Ans = "": METHOD = %False                                ' Return Null and fail
      END IF                                                      '
      me.SelEnd()                                                 ' Always end the select
   END METHOD                                                     '

   METHOD RowDelete(sKey AS STRING) AS LONG                       '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a single row                                                                          |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("delete from DataTable where DBKey = '" + UCASE$(sKey) + "'")   '
   END METHOD                                                     '

   METHOD SelBegin(sql AS STRING) AS LONG                         '
   '-----------------------------------------------------------------------------------------------+
   '- Execute a SQLquery and return a record set                                                   |
   '-----------------------------------------------------------------------------------------------+
   LOCAL i, RC AS LONG                                            '
   LOCAL lclErrorSz  AS LONG                                      '
   LOCAL lclCol AS ASCIIZ PTR                                     '
      RESET TablePtr, tRowCount, tColCount, tRowNum, tFirstCol    '
      ERASE tColumns()                                            '
      IF LEN(sql) THEN                                            '
         IF sqlite_get_table(hDB, BYVAL STRPTR(sql), TablePtr, tRowCount, tColCount, lclErrorSz) = 0 THEN   '
            IF tRowCount = 0 THEN                                 ' Anything come back?
               METHOD = %True                                     ' No, just return True
               sqlite_free_table TablePtr                         ' Free the table
            ELSE                                                  ' Something came back
               REDIM tColumns(1 TO tColCount)                     ' Redim tableto fit
               FOR i = 1 TO tColCount                             ' Populate it
                  lclCol = @TablePtr[i - 1]                       '
                  tColumns(i) = @lclCol                           '
               NEXT i                                             '
               METHOD = %True                                     ' Return True
            END IF                                                '
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelColumn(columnNo AS LONG) AS STRING                   '
   '-----------------------------------------------------------------------------------------------+
   '- Get column name for columnNo in SQLquery results, ONE based index                            |
   '-----------------------------------------------------------------------------------------------+
      IF columnNo > 0 AND columnNo <= UBOUND(tColumns) THEN METHOD = tColumns(columnNo)   '
   END METHOD                                                     '

   METHOD SelColumnNo(sColumn AS STRING) AS LONG                  '
   '-----------------------------------------------------------------------------------------------+
   '- Return position of column in SQLquery results, zero if not found                             |
   '-----------------------------------------------------------------------------------------------+
   LOCAL x AS LONG                                                ' Local Data
      ARRAY SCAN tColumns(), COLLATE UCASE, = sColumn, TO x       ' Search for it
      METHOD = x                                                  ' Return answer
   END METHOD                                                     '

   METHOD SelEnd()                                                '
   '-----------------------------------------------------------------------------------------------+
   '- Close SQLquery                                                                               |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN sqlite_free_table TablePtr                ' If we got something, Free it
      RESET TablePtr, tRowCount, tColCount, tRowNum, tFirstCol    ' Clear the data fields
      ERASE tColumns()                                            '
   END METHOD                                                     '

   METHOD SelFirst() AS LONG                                      '
   '-----------------------------------------------------------------------------------------------+
   '- Move to first row in SQLquery results                                                        |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         tRowNum = 1                                              ' Set to Row 1
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGet(sColumn AS STRING) AS STRING                     '
   '-----------------------------------------------------------------------------------------------+
   '- Get value for column on current row of SQLquery results                                      |
   '-----------------------------------------------------------------------------------------------+
   LOCAL x, ndx, UTFError AS LONG, ANSIAnswer AS STRING           ' Local data
   LOCAL lclCol AS ASCIIZ PTR                                     '
      IF tRowNum > 0 AND tRowNum <= tRowCount THEN                ' If on a valid Row number
         x = me.SelColumnNo(sColumn)                              ' Get position of column in results
         ndx = tFirstCol + x - 1                                  ' Calc Index
         lclCol = @TablePtr[ndx]                                  ' Pick up column
         ANSIAnswer = @lclCol                                     ' ASCIIZ to STRING
         METHOD = StrUtf82Ansi(ANSIAnswer, UTFError)              ' Return the string
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGetAt(columnNo AS LONG) AS STRING                    '
   '-----------------------------------------------------------------------------------------------+
   '- Get value for columnNo on current row of SQLquery results                                    |
   '-----------------------------------------------------------------------------------------------+
    LOCAL ndx, UTFError AS LONG, ANSIAnswer AS STRING             ' Local Data
    LOCAL lclCol AS ASCIIZ PTR                                    '
      IF tRowNum > 0 AND tRowNum <= tRowCount THEN                ' If valid Row number
         IF columnNo > 0 AND columnNo <= UBOUND(tColumns) THEN    ' and valid Column number
            ndx = tFirstCol + columnNo - 1                        ' Calc index
            lclCol = @TablePtr[ndx]                               ' Pick up the column
            ANSIAnswer = @lclCol                                  ' ASCIIZ to STRING
         METHOD = StrUtf82Ansi(ANSIAnswer, UTFError)              ' Return the string
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelGoTo(srow AS LONG) AS LONG                           '
   '-----------------------------------------------------------------------------------------------+
   '- Move to row in SQLquery results                                                              |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount AND srow > 0 AND srow <= tRowCount THEN        ' If requested row is valid
         tRowNum = srow                                           ' Go to it
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelLast() AS LONG                                       '
   '-----------------------------------------------------------------------------------------------+
   '- Move to last row in SQLquery results                                                         |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         tRowNum = tRowCount                                      ' Set to last row
         tFirstCol = tRowNum * tColCount                          ' Set FirstCol
         METHOD = %True                                           ' Return True
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelNext() AS LONG                                       '
   '-----------------------------------------------------------------------------------------------+
   '- Move to next row in SQLquery results                                                         |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         INCR tRowNum                                             ' Bump Row number
         IF tRowNum > 0 AND tRowNum <= tRowCount THEN             ' If not past the end
            tFirstCol = tRowNum * tColCount                       ' Set FirstCol
            METHOD = %True                                        ' Return True
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SelPrevious() AS LONG                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Move to previous row in SQLquery results                                                     |
   '-----------------------------------------------------------------------------------------------+
      IF tRowCount THEN                                           ' If we even have a Row count
         DECR tRowNum                                             ' Drop by 1
         IF tRowNum > 0 AND tRowNum <= tRowCount THEN             ' If not too low
            tFirstCol = tRowNum * tColCount                       ' Set FirstCol
            METHOD = %True                                        ' Return True
         END IF                                                   '
      END IF                                                      '
   END METHOD                                                     '

   METHOD SetString(sKey AS STRING, sStr AS STRING) AS LONG       '
   '-----------------------------------------------------------------------------------------------+
   '- Add/Replace a Key/Data pair to the Config DB                                                 |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("insert or replace into DataTable values ('" + UCASE$(sKey) + "', '" + me.Fix(sStr) + "')")   '
   END METHOD                                                     '

   METHOD TableClear() AS LONG                                    '
   '-----------------------------------------------------------------------------------------------+
   '- Clear a Table                                                                                |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("delete from DataTable")                ' Clear all rows
   END METHOD                                                     '

   METHOD TableCreate() AS LONG                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Create a Db Table                                                                            |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("create table if not exists DataTable" + _ '
                    " (DBKey text primary key unique collate nocase, DBData text)") '
   END METHOD                                                     '

   METHOD TableDelete() AS LONG                                   '
   '-----------------------------------------------------------------------------------------------+
   '- Delete a Table                                                                               |
   '-----------------------------------------------------------------------------------------------+
      METHOD = me.Execute("drop table if exists DataTable")       ' Kill the table
   END METHOD                                                     '

   METHOD TableExist() AS LONG                                    '
   '-----------------------------------------------------------------------------------------------+
   '- See if a Table exists                                                                        |
   '-----------------------------------------------------------------------------------------------+
   LOCAL lclName AS STRING                                        '
      IF me.Quickie("select name FROM sqlite_master WHERE type ='table' AND name = DataTable", lclName) THEN   '
         METHOD = %True                                           ' Say True if OK
      END IF                                                      '
   END METHOD                                                     '

   METHOD UpdateString(sKey AS STRING, sStr AS STRING) AS LONG    '
   '-----------------------------------------------------------------------------------------------+
   '- Update a table value                                                                         |
   '-----------------------------------------------------------------------------------------------+
       METHOD = me.execute("update DataTable SET DBData = '" + me.Fix(sStr) + "' where DBKey = '" + UCASE$(sKey) + "'") '
   END METHOD                                                     '

   END INTERFACE                                                  '
END CLASS                                                         '
